/*
  Top10, a racing simulator
  Copyright (C) 2000-2007  Johann Deneux
  
  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.
  
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
  
  Authors can be contacted at following electronic addresses:
  Johann Deneux: johann.deneux@gmail.com
*/

#include "ArcToolDlg.hh"
#include <QtGui/QMessageBox>
#include "math/Vec2D.hh"

ArcToolDlg::ArcToolDlg(QWidget* parent)
: QDialog(parent),
  path_ed(0),
  listener(0)
{
  setupUi(this);
}

ArcToolDlg::~ArcToolDlg()
{
  if (listener && path_ed)
    path_ed->removeListener(listener);
}

void ArcToolDlg::setWaypointsEditor(top10::tracked::PathEditor* ed)
{
  path_ed = ed;

  NotSelected_listWidget->clear();
  Selected_listWidget->clear();

  for (std::vector<top10::tracked::PathEditor::WayPoint>::const_iterator it = ed->waypoints.begin();
    it != ed->waypoints.end();
    ++it)
  {
    newWaypoint(*it);
  }

  if (listener)
    path_ed->removeListener(listener);

  listener = new Listener(this);
  path_ed->addListener(listener);
}

void ArcToolDlg::newWaypoint(const top10::tracked::PathEditor::WayPoint& wp)
{
  QListWidgetItem* new_item = new QListWidgetItem;
  new_item->setData(Qt::UserRole, wp.id);
  new_item->setText(wp.name.c_str());

  NotSelected_listWidget->addItem(new_item);
}

void ArcToolDlg::delWaypoint(const top10::tracked::PathEditor::WayPoint wp)
{

  for (int i=0; i<NotSelected_listWidget->count(); ++i)
  {
    if (NotSelected_listWidget->item(i)->data(Qt::UserRole).toInt() == wp.id)
    {
      QListWidgetItem *found = NotSelected_listWidget->takeItem(i);
      delete found;
      --i;
    }
  }

  for (int i=0; i<Selected_listWidget->count(); ++i)
  {
    if (Selected_listWidget->item(i)->data(Qt::UserRole).toInt() == wp.id)
    {
      QListWidgetItem *found = Selected_listWidget->takeItem(i);
      delete found;
      --i;
    }
  }

}

void ArcToolDlg::renameWaypoint(const top10::tracked::PathEditor::WayPoint& wp)
{

  for (int i=0; i<NotSelected_listWidget->count(); ++i)
  {
    if (NotSelected_listWidget->item(i)->data(Qt::UserRole).toInt() == wp.id)
    {
      NotSelected_listWidget->item(i)->setText(wp.name.c_str());
    }
  }

  for (int i=0; i<Selected_listWidget->count(); ++i)
  {
    if (Selected_listWidget->item(i)->data(Qt::UserRole).toInt() == wp.id)
    {
      Selected_listWidget->item(i)->setText(wp.name.c_str());
    }
  }

}

void ArcToolDlg::on_Add_pushButton_pressed()
{

  QListWidgetItem* moved_item = NotSelected_listWidget->takeItem(NotSelected_listWidget->currentRow());
  if (moved_item)
    Selected_listWidget->addItem(moved_item);

}

void ArcToolDlg::on_Remove_pushButton_pressed()
{

  QListWidgetItem* moved_item = Selected_listWidget->takeItem(Selected_listWidget->currentRow());
  if (moved_item)
    NotSelected_listWidget->addItem(moved_item);

}

void ArcToolDlg::on_OK_pushButton_pressed()
{
  if (Selected_listWidget->count() < 3)
  {
    QMessageBox::information(this, tr("Error"), tr("You must select at least 3 waypoints."), QMessageBox::Ok);
    return;
  }

  bool conv_ok = true;
  double radius = Radius_lineEdit->text().toDouble(&conv_ok);

  if (!conv_ok)
  {
    QMessageBox::information(this, tr("Error"), tr("The radius must be a number."), QMessageBox::Ok);
    return;
  }

  double x, z;
  if (!computeCenter(radius, x,z))
    return;

  moveWaypoints(radius, x,z);  
}

void ArcToolDlg::moveWaypoints(double radius, double x, double z)
{
  double alt_start, alt_end, alt_incr;
  int idx = path_ed->findById(Selected_listWidget->item(0)->data(Qt::UserRole).toInt());
  alt_start = path_ed->waypoints.at(idx).pos.y;
  
  idx = path_ed->findById(Selected_listWidget->item(Selected_listWidget->count() -1)->data(Qt::UserRole).toInt());
  alt_end = path_ed->waypoints.at(idx).pos.y;

  alt_incr = (alt_end - alt_start) / (Selected_listWidget->count()-1);

  top10::math::Vector center(x, alt_start, z);
  double alt = alt_start;

  for (int i=0; i<Selected_listWidget->count(); ++i)
  {
    int idx = path_ed->findById(Selected_listWidget->item(i)->data(Qt::UserRole).toInt());
    path_ed->gotoIdx(idx);
    
    top10::math::Vector pos = path_ed->getCurrentPos();
    top10::math::Vector diff = pos - center;
    diff.y = 0;
    
    double s = diff.size();
    if (s > SMALL_VECTOR)
    {
      diff /= s;
      top10::math::Vector tg(-diff.z, 0.0, diff.x);

      diff = radius * diff;
      if (i != 0 && i != Selected_listWidget->count())
      {
	top10::math::Vector new_pos = center + diff;
	new_pos.y = alt;
	path_ed->setCurrentPos(new_pos);
      }

      path_ed->setCurrentTangent(tg);
    }

    alt += alt_incr;
  }
}

bool ArcToolDlg::computeCenter(double radius, double& x, double &z)
{
  using top10::math::Vec2D;

  int idx = path_ed->findById(Selected_listWidget->item(0)->data(Qt::UserRole).toInt());
  Vec2D A(path_ed->waypoints.at(idx).pos.z, path_ed->waypoints.at(idx).pos.x);

  idx = path_ed->findById(Selected_listWidget->item(Selected_listWidget->count() -1)->data(Qt::UserRole).toInt());
  Vec2D B(path_ed->waypoints.at(idx).pos.z, path_ed->waypoints.at(idx).pos.x);

  Vec2D AB = B-A;
  Vec2D AI = 0.5*AB;
  
  Vec2D IX(AB.y, -AB.x); // rotate 90 deg.

  // Normalize
  double s = IX.size();
  if (s <= SMALL_VECTOR)
  {
    QMessageBox::information(this, tr("Error"), tr("The first and last selected waypoints are too close."), QMessageBox::Ok);
    return false;
  }
  IX /= s;

  // Adjust length to match radius
  s = radius*radius - AI.size2();
  if (s < 0)
  {
    QMessageBox::information(this, tr("Error"), tr("The radius is too small"), QMessageBox::Ok);
    return false;
  }

  s = sqrt(s);
  IX = s*IX;

  assert(fabs((AI+IX).size() - radius) < 0.1);

  IX = A + AI + IX;
  x = IX.y;
  z = IX.x;

  return true;
}

ArcToolDlg::Listener::Listener(ArcToolDlg* d)
: tool(d)
{
}

void ArcToolDlg::Listener::notifyAdd(const top10::tracked::PathEditor::WayPoint& wp)
{
  tool->newWaypoint(wp);
}

void ArcToolDlg::Listener::notifyRemove(top10::tracked::PathEditor::WayPoint wp)
{
  tool->delWaypoint(wp);
}

void ArcToolDlg::Listener::notifyName(const top10::tracked::PathEditor::WayPoint& wp)
{
  tool->renameWaypoint(wp);
}
