/*
  Top10, a racing simulator
  Copyright (C) 2000-2005  Johann Deneux
  
  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.
  
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
  
  Authors can be contacted at following electronic addresses:
  Johann Deneux: johann.deneux@it.uu.se
*/

#include "StartingAreaDialog.hh"
#include "DialogManager.hh"
#include <cassert>

/*
 * Callbacks
 */
/* This is a template callback. The template parameters are the affected point  (1 or 2),
   and the component (0:X, 1:Y, Z:2)
*/
template<int pointN, int coordN>
static void inputCB(puObject* ob)
{
  assert(pointN == 1 || pointN == 2);
  assert(coordN >= 0 && coordN < 3);

  StartingAreaDialog* dialog = (StartingAreaDialog*)ob->getUserData();

  // Get the point
  top10::math::Vector pt;
  if (pointN == 2) pt = dialog->getManager()->sa_ed->starting_area.getPoint2();
  else if (pointN == 1) pt = dialog->getManager()->sa_ed->starting_area.getPoint1();
  else assert(false);

  // Retrieve the input's value and modify the component
  float tmp;
  ob->getValue(&tmp);
  pt[coordN] = tmp;

  // Set the point
  if (pointN == 2) dialog->getManager()->sa_ed->starting_area.setPoint2(pt);
  else if (pointN == 1) dialog->getManager()->sa_ed->starting_area.setPoint1(pt);  

  dialog->getManager()->sa_ed->setDirection();
}

template<int pointN>
static void pickCB(puObject* ob)
{
  assert(pointN == 1 || pointN == 2);
  StartingAreaDialog* dialog = (StartingAreaDialog*)ob->getUserData();

  top10::math::Vector pt;
  if (pointN == 1)
    dialog->getManager()->sa_ed->pickLeft();
  else if (pointN == 2)
    dialog->getManager()->sa_ed->pickRight();
  else assert(false);

  dialog->getManager()->sa_ed->setDirection();

  dialog->remove();
  dialog->install();
  dialog->show();
}

static void swapCB(puObject* ob)
{
  StartingAreaDialog* dialog = (StartingAreaDialog*)ob->getUserData();

  dialog->getManager()->sa_ed->swap();
  dialog->getManager()->sa_ed->setDirection();

  dialog->remove();
  dialog->install();
  dialog->show();
}

/*
 * Class implementation
 */
StartingAreaDialog::StartingAreaDialog(DialogManager* manager):
  Dialog(manager, "Starting Area")
{
}

void StartingAreaDialog::populate()
{
  top10::math::Vector pt = getManager()->sa_ed->starting_area.getPoint2();

  puInput* pt2_x = new puInput(0, H, M_W, 2*H);
  pt2_x->setUserData(this);
  pt2_x->setCallback(inputCB<2, 0>);
  pt2_x->setValue((float)pt.x);

  puInput* pt2_y = new puInput(0, 2*H, M_W, 3*H);
  pt2_y->setUserData(this);
  pt2_y->setCallback(inputCB<2, 1>);
  pt2_y->setValue((float)pt.y);

  puInput* pt2_z = new puInput(0, 3*H, M_W, 4*H);
  pt2_z->setUserData(this);
  pt2_z->setCallback(inputCB<2, 2>);
  pt2_z->setValue((float)pt.z);

  pt = getManager()->sa_ed->starting_area.getPoint1();
  puInput* pt1_x = new puInput(0, 4.5*H, M_W, 5.5*H);
  pt1_x->setUserData(this);
  pt1_x->setCallback(inputCB<1, 0>);
  pt1_x->setValue((float)pt.x);

  puInput* pt1_y = new puInput(0, 5.5*H, M_W, 6.5*H);
  pt1_y->setUserData(this);
  pt1_y->setCallback(inputCB<1, 1>);
  pt1_y->setValue((float)pt.y);

  puInput* pt1_z = new puInput(0, 6.5*H, M_W, 7.5*H);
  pt1_z->setUserData(this);
  pt1_z->setCallback(inputCB<1, 2>);
  pt1_z->setValue((float)pt.z);

  puOneShot* pick2 = new puOneShot(M_W +5, H, M_W+5 +M_W, 4*H);
  pick2->setLegend("Pick");
  pick2->setUserData(this);
  pick2->setCallback(pickCB<2>);

  puOneShot* pick1 = new puOneShot(M_W +5, 4.5*H, M_W+5 +M_W, 7.5*H);
  pick1->setLegend("Pick");
  pick1->setUserData(this);
  pick1->setCallback(pickCB<1>);

  puOneShot* swap = new puOneShot(2*M_W +10, 3*H, 2*M_W+10 +M_W, 5.5*H);
  swap->setLegend("Swap");
  swap->setUserData(this);
  swap->setCallback(swapCB);
}
