#include "WallEditor.hh"
#include "math/Hermite.hh"
#include "graphX/Extruder.hh"
#include "graphX/MaterialNode.hh"
#include "graphX/MeshMarker.hh"
#include "graphX/TextureNode.hh"
#include "graphX/TextureTransform.hh"
#include "graphX/TextureManager.hh"
#include "util/NameFinder.hh"
#include "util/findpred.hh"

namespace top10
{
  namespace tracked
  {
    const char* WallEditor::NODE_NAME = "walls";
    const int   WallEditor::WALL_TYPE = 0;
    const int   WallEditor::MESH_TYPE = 1;

    WallEditor::WallEditor()
      : Drawable(NODE_NAME),
      m_path_editor(0)
    {
      clearState();
      m_listener = new Listener(this);
    }



    void WallEditor::updateNode()
    {
      using namespace top10::graphX;
      Node* new_node = generate(true /*mark_current*/);

      removeChild( m_node.getPtr() );
      m_node = new_node;
      addChild( m_node.getPtr() );
    }



    top10::graphX::GroupNode* WallEditor::generate(bool mark_current) const
    {
      using namespace top10::graphX;

      MaterialNode* mat = new MaterialNode;
      mat->r = mat->g = mat->b = 255;

      top10::util::Ref<MeshMarker> marker(new MeshMarker);
      if (mark_current)
      {
	mat->addChild(marker.getPtr());
      }

      MeshNodeRefs meshes;
      int i=0;
      for (std::vector<Wall>::const_iterator it = m_walls.begin(); it != m_walls.end(); ++it, ++i)
      {
	Node* wall = generateWall(*it);

	if (wall)
	{
	  Node* child = wall;

	  TextureTransform* texture_trans = 0;
	  if (!it->m_texture.getFilename().empty())
	  {
	    texture_trans = new TextureTransform;
	    top10::math::Matrix4 T;
	    T(0,0) = it->m_texture.getScaleLat();
	    T(1,1) = it->m_texture.getScaleLong();
	    T(2,2) = 1.0;
	    T(3,3) = 1.0;
	    T(0, 3) = it->m_texture.getTranslateLat();
	    T(1,3) = it->m_texture.getTranslateLong();
	    texture_trans->setToWorld(T);

	    TextureNode* texture = new TextureNode;
	    texture_trans->addChild(texture);
	    texture->setTextureId(TextureManager::getInstance()->getTexture( it->m_texture.getFilename() ));
	    texture->addChild(wall);

	    child = texture_trans;
	  }

	  mat->addChild(child);

	  if (mark_current && i == m_current)
	    marker->addChild(child);
	}
      }

      return mat;
    }



    top10::graphX::GroupNode*
      WallEditor::generateWall(const Wall& wall) const
    {
      if (wall.m_wp_ids.size() <= 1)
	return 0;

      std::vector< top10::math::Vec2D > shape;
      generateWallShape(wall, &shape);

      top10::graphX::GroupNode* grp = new top10::graphX::GroupNode;

      int idx1 = m_path_editor->findById(wall.m_wp_ids[0]);
      int idx2;
      for (std::vector<int>::const_iterator it = wall.m_wp_ids.begin() +1; it != wall.m_wp_ids.end(); ++it)
      {
	idx2 = m_path_editor->findById(*it);

	const PathEditor::WayPoint pt1 = m_path_editor->waypoints.at(idx1);
	const PathEditor::WayPoint pt2 = m_path_editor->waypoints.at(idx2);

	top10::math::Hermite curve;
	curve.setEnd1(pt1.pos);
	curve.setTangent1(pt1.tg);
	curve.setEnd2(pt2.pos);
	curve.setTangent2(pt2.tg);

	top10::graphX::Node* node = top10::graphX::extrude(curve, shape);
	grp->addChild(node);

	idx1 = idx2;
      }

      return grp;
    }



    void WallEditor::generateWallShape(const Wall& wall, std::vector< top10::math::Vec2D >* shape_out)
    {
      using top10::math::Vec2D;

      assert( wall.m_thickness >= 0.0 );
      assert( wall.m_height > 0.0 );

      shape_out->clear();

      if (wall.m_type == Wall::WALL)
      {
	shape_out->push_back( Vec2D(0.0, 0.0) );
	shape_out->push_back( Vec2D(0.0, wall.m_height) );
	shape_out->push_back( Vec2D(-wall.m_thickness, wall.m_height + wall.m_thickness * tan(M_PI * wall.m_slant / 180.0)) );
	shape_out->push_back( Vec2D(-wall.m_thickness, 0.0) );
	shape_out->push_back( shape_out->front() );
      }
      else if (wall.m_type == Wall::MESH)
      {
	shape_out->push_back( Vec2D(0.0, 0.0) );
	shape_out->push_back( Vec2D(0.0, wall.m_height) );
	shape_out->push_back( Vec2D(-wall.m_thickness * tan(M_PI * wall.m_slant / 180.0), wall.m_height + wall.m_thickness) );
      }
    }



    void WallEditor::setPathEditor(top10::tracked::PathEditor* ed)
    {
      if (m_path_editor)
	m_path_editor->removeListener(m_listener.getPtr());

      m_path_editor = ed;
      m_path_editor->addListener(m_listener.getPtr());
    }



    void WallEditor::clearState()
    {
      m_walls.clear();
      m_current = -1;
      updateNode();
    }



    int WallEditor::loadXml(const TiXmlElement* node)
    {
      bool status = true;

      const TiXmlElement* child = node->FirstChildElement("wall");
      while (child)
      {
	Wall new_one;

	status = child->QueryDoubleAttribute("height", &new_one.m_height) == 0 && status;
	status = child->QueryDoubleAttribute("slant", &new_one.m_slant) == 0 && status;
	status = child->QueryDoubleAttribute("thickness", &new_one.m_thickness) == 0 && status;

	int wall_type = WALL_TYPE;
	status = child->QueryIntAttribute("type", &wall_type) == 0 && status;
	if (wall_type == WALL_TYPE)
	  new_one.m_type = Wall::WALL;
	else if (wall_type == MESH_TYPE)
	  new_one.m_type = Wall::MESH;

	const char* name = child->Attribute("name");
	if (name)
	  new_one.m_name = name;
	status = !!name && status;

	const TiXmlElement* ids = child->FirstChildElement("id");
	while (ids)
	{
	  int val;
	  bool s = ids->QueryIntAttribute("val", &val) == 0;
	  if (s)
	    new_one.m_wp_ids.push_back(val);
	  else
	    status = false;

	  ids = ids->NextSiblingElement("id");
	}

	const TiXmlElement* texture = child->FirstChildElement(new_one.m_texture.getNodeName());
	if (texture)
	  new_one.m_texture.load(texture);

	m_walls.push_back(new_one);

	child = child->NextSiblingElement("wall");
      }

      m_current = 0;

      callClient();
      updateNode();

      return status ? 0 : -1;
    }



    int WallEditor::saveXml(TiXmlElement* node) const
    {
      for (std::vector<Wall>::const_iterator it = m_walls.begin(); it != m_walls.end(); ++it)
      {
	TiXmlElement child("wall");
	child.SetDoubleAttribute("height", it->m_height);
	child.SetDoubleAttribute("slant", it->m_slant);
	child.SetDoubleAttribute("thickness", it->m_thickness);
	child.SetAttribute("name", it->m_name);
	child.SetAttribute("type", it->m_type == Wall::WALL ? WALL_TYPE : MESH_TYPE);

	for (std::vector<int>::const_iterator id_it = it->m_wp_ids.begin(); id_it != it->m_wp_ids.end(); ++id_it)
	{
	  TiXmlElement id_node("id");
	  id_node.SetAttribute("val", *id_it);
	  child.InsertEndChild(id_node);
	}

	TiXmlElement texture(it->m_texture.getNodeName());
	it->m_texture.save(&texture);
	child.InsertEndChild(texture);

	node->InsertEndChild(child);
      }

      return 0;
    }

 

    std::string WallEditor::getOrigin() const
    {
      return "WallEditor";
    }



    std::string WallEditor::newWall()
    {
      Wall new_one;
      
      new_one.m_height = 2.0;
      new_one.m_slant = 0.0;
      new_one.m_thickness = 0.2;
      new_one.m_name = top10::util::NameFinder< top10::util::Find_M_Named >::getNewName(m_walls.begin(), m_walls.end(), "Unnamed");
      new_one.m_type = Wall::WALL;

      m_walls.push_back(new_one);

      m_current = m_walls.size() -1;

      updateNode();

      return new_one.m_name;
    }



    bool WallEditor::renameWall(const std::string& new_name)
    {
      if (m_current >= 0 && (unsigned int)m_current < m_walls.size())
      {
	top10::util::Find_M_Named pred(new_name);
	if (std::find_if(m_walls.begin(), m_walls.end(), pred) == m_walls.end())
	{
	  m_walls.at(m_current).m_name = new_name;
	  return true;
	}
      }

      return false;
    }



    bool WallEditor::deleteWall()
    {
      if (m_current >= 0 && (unsigned int)m_current < m_walls.size())
      {
	m_walls.erase(m_walls.begin() + m_current);

	updateNode();

	return true;
      }
      return false;
    }



    bool WallEditor::setCurrentWall(const std::string& name)
    {
      int i=0;
      for (; (unsigned int)i<m_walls.size() && m_walls.at(i).m_name != name; ++i);
      
      if ((unsigned int)i < m_walls.size())
      {
	m_current = i;

	updateNode();

	return true;
      }

      return false;
    }



    void WallEditor::setTypeWall()
    {
      if (m_current >= 0 && (unsigned int)m_current < m_walls.size())
	m_walls.at(m_current).m_type = Wall::WALL;
    }



    void WallEditor::setTypeMesh()
    {
      if (m_current >= 0 && (unsigned int) m_current < m_walls.size())
	m_walls.at(m_current).m_type = Wall::MESH;
    }



    void WallEditor::getWallNames(std::list<std::string>* walls) const
    {
      for (std::vector<Wall>::const_iterator it = m_walls.begin(); it != m_walls.end(); ++it)
      {
	walls->push_back(it->m_name);
      }
    }



    std::string WallEditor::getWallName() const
    {
      if (m_current >= 0 && (unsigned int)m_current < m_walls.size())
	return m_walls.at(m_current).m_name;
      else
	return std::string();
    }




    void WallEditor::getAllPointNames(std::list< std::string >* names) const
    {
      if (!m_path_editor)
	return;

      for (std::vector<PathEditor::WayPoint>::const_iterator it = m_path_editor->waypoints.begin();
	   it != m_path_editor->waypoints.end();
	   ++it)
      {
	names->push_back(it->name);
      }
    }




    bool WallEditor::setHeight(double d)
    {
      if (d > 0.0 && m_current >= 0 && (unsigned int)m_current <= m_walls.size())
      {
	m_walls.at(m_current).m_height = d;

	updateNode();

	return true;
      }

      return false;
    }



    bool WallEditor::setThickness(double d)
    {
      if (d > 0.0 && m_current >= 0 && (unsigned int)m_current <= m_walls.size())
      {
	m_walls.at(m_current).m_thickness = d;

	updateNode();

	return true;
      }

      return false;
    }



    bool WallEditor::setSlant(double d)
    {
      if (m_current >= 0 && (unsigned int)m_current <= m_walls.size())
      {
	m_walls.at(m_current).m_slant = d;

	updateNode();

	return true;
      }

      return false;
    }


    bool WallEditor::insertPoint(int pos, const std::string& wp_name)
    {
      int wp_idx = m_path_editor->findNamed(wp_name);
      if (wp_idx == -1)
	return false;

      if (m_current >= 0 && (unsigned int)m_current < m_walls.size())
      {
	std::vector<Wall>::iterator wall_it = m_walls.begin() + m_current;
	
	if (pos >= -1 && pos < (int)wall_it->m_wp_ids.size())
	{
	  std::vector<int>::iterator insert_pos;

	  if (pos == -1)
	    insert_pos = m_walls.at(m_current).m_wp_ids.end();
	  else
	    insert_pos = m_walls.at(m_current).m_wp_ids.begin() + pos;

	  wall_it->m_wp_ids.insert(insert_pos, m_path_editor->waypoints.at(wp_idx).id);

	  updateNode();

	  return true;
	}
      }

      return false;
    }



    bool WallEditor::removePoint(const std::string& name)
    {
      if (m_current >= 0 && (unsigned int)m_current < m_walls.size())
      {
	std::vector<Wall>::iterator wall_it = m_walls.begin() + m_current;
	
	int idx = m_path_editor->findNamed(name);
	if (idx != -1)
	{
	  int uid = m_path_editor->waypoints.at(idx).id;
	  std::vector<int>::iterator find_it = std::find(wall_it->m_wp_ids.begin(), wall_it->m_wp_ids.end(), uid);

	  if (find_it != wall_it->m_wp_ids.end())
	  {
	    wall_it->m_wp_ids.erase(find_it);

	    updateNode();

	    return true;
	  }
	}
      }

      return false;
    }



    bool WallEditor::isTypeWall() const
    {
      if (m_current >= 0 && (unsigned int)m_current < m_walls.size())
	return m_walls.at(m_current).m_type == Wall::WALL;
      return false;
    }



    bool WallEditor::isTypeMesh() const
    {
      if (m_current >= 0 && (unsigned int)m_current < m_walls.size())
	return m_walls.at(m_current).m_type == Wall::MESH;
      return false;
    }



    std::string WallEditor::getName() const
    {
      if (m_current >= 0 && (unsigned int)m_current < m_walls.size())
	return m_walls.at(m_current).m_name;

      return "";
    }



    double WallEditor::getHeight() const
    {
      if (m_current >= 0 && (unsigned int)m_current < m_walls.size())
	return m_walls.at(m_current).m_height;

      return 0.0;
    }



    double WallEditor::getThickness() const
    {
      if (m_current >= 0 && (unsigned int)m_current < m_walls.size())
	return m_walls.at(m_current).m_thickness;

      return 0.0;
    }



    double WallEditor::getSlant() const
    {
      if (m_current >= 0 && (unsigned int)m_current < m_walls.size())
	return m_walls.at(m_current).m_slant;

      return 0.0;
    }



    void WallEditor::getPointNames(std::list< std::string >* out)
    {
      if (m_current >= 0 && (unsigned int)m_current < m_walls.size())
      {
	for (std::vector<int>::const_iterator it = m_walls.at(m_current).m_wp_ids.begin(); it != m_walls.at(m_current).m_wp_ids.end(); ++it)
	{
	  int idx = m_path_editor->findById(*it);
	  if (idx != -1)
	    out->push_back(m_path_editor->waypoints.at(idx).name);
	}
      }
    }



    top10::track::TextureSpec* WallEditor::getTexture()
    {
      if (m_current >= 0 && (unsigned int)m_current < m_walls.size())
      {
	return & m_walls.at(m_current).m_texture;
      }

      return 0;
    }



    /*
     * WallEditor::Listener
     */
    WallEditor::Listener::Listener(top10::tracked::WallEditor* ed)
      : m_editor(ed)
    {
    }



    void WallEditor::Listener::notifyRemove(top10::tracked::PathEditor::WayPoint wp)
    {
      for (std::vector< Wall >::iterator it = m_editor->m_walls.begin(); it != m_editor->m_walls.end(); ++it)
      {
	std::vector<int> tmp;
	for (unsigned int i=0; i<it->m_wp_ids.size(); ++i)
	{
	  if (it->m_wp_ids[i] != wp.id)
	    tmp.push_back(it->m_wp_ids[i]);
	}
	tmp.swap(it->m_wp_ids);
      }

      m_editor->callClient();
      m_editor->updateNode();
    }



    void WallEditor::Listener::notifyMove(const top10::tracked::PathEditor::WayPoint& wp)
    {
      m_editor->callClient();
      m_editor->updateNode();
    }
  }
}
