/*
  Top 10, a racing simulator
  Copyright (C) 2003,2005,2006  Johann Deneux
  
  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.
  
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
  
  Authors can be contacted at following electronic addresses:
  Johann Deneux: johann.deneux@gmail.com
*/
#include "TriangulationEditor.hh"
#include "graphX/MeshNode.hh"
#include "graphX/WireNode.hh"
#include "graphX/Wireframe.hh"

namespace top10 {
  
namespace math {
#include "math/Triangulation-template.cpp"
}

namespace tracked {

using top10::util::Log;
  
TriangulationEditor::TriangulationEditor():
  top10::tracked::Drawable("triangulation"),
  sections_ed(0),
  max_angle(5),
  min_dist(0.25)
{
  ground_mat = new top10::graphX::MaterialNode;
  ground_mat->r = ground_mat->g = ground_mat->b = 255;
  addChild(ground_mat.getPtr());
  
  error_mat = new top10::graphX::MaterialNode;
  error_mat->r = 255; error_mat->g = error_mat->b = 128;
  addChild(error_mat.getPtr());

  clearState();
}

void TriangulationEditor::clearState()
{
  grass_texture.clearState();
  side_texture.clearState();
  grass_distance = 1.0;
}

std::string TriangulationEditor::getOrigin() const
{
  return "TriangulationEditor";
}

void TriangulationEditor::setSectionsEditor(SectionsEditor* p)
{
  sections_ed = p;
}

void TriangulationEditor::updateView()
{
  if (!sections_ed) return;

  top10::track::SectionGraph* sg = 0;
  try {
    while (ground_mat->getChild(0)) ground_mat->removeChild(ground_mat->getChild(0));
    while (error_mat->getChild(0)) error_mat->removeChild(error_mat->getChild(0));

    // Prepare to triangulate
    sg = sections_ed->makeSectionGraph();
    sg->setTextures(grass_distance, &grass_texture, &side_texture);
    sg->setMaxAngle(max_angle);
    sg->setMinDist(min_dist);

    // Triangulate
    top10::util::Ref<top10::graphX::GroupNode> ground(sg->getGround());
    for_all_children(ground, node)
    {
      ground_mat->addChild(node);
    }
    for_all_children_end;

    // Debug: show the outline
#if 0
    std::list< std::vector<top10::track::TexturedVertex> > outlines = sg->getOutlines();
    for (std::list< std::vector<top10::track::TexturedVertex> >::const_iterator out_it = outlines.begin();
      out_it != outlines.end();
      ++out_it)
    {
      top10::math::Vector prev = out_it->front().pos;
      for (std::vector<top10::track::TexturedVertex>::const_iterator v_it = out_it->begin();
	v_it != out_it->end();
	++v_it)
      {
	top10::graphX::WireNode* mark1 = new top10::graphX::WireNode;
	mark1->addWire(v_it->pos - top10::math::Vector(-0.3, 0.05, -0.3), 
	  v_it->pos + top10::math::Vector(-0.3, 0.05, -0.3));
	mark1->addWire(v_it->pos - top10::math::Vector(-0.3, 0.05, +0.3), 
	  v_it->pos + top10::math::Vector(-0.3, 0.05, +0.3));

	mark1->addWire(v_it->pos, prev);

	prev = v_it->pos;
	error_mat->addChild(mark1);
      }
    }
#endif
  }
  catch(top10::track::TrackTriangulation::ConflictingConstraint& e)
  {    
    top10::graphX::WireNode* mark1 = new top10::graphX::WireNode;
    mark1->addWire(e.c1v1.pos, e.c1v2.pos);
    mark1->addWire(e.c2v1.pos, e.c2v2.pos);

    error_mat->addChild(mark1);

    delete sg; sg = 0;

    std::ostringstream buf;
    buf << "Failed to build triangulation. Reason is: " << e <<". ";

    Log::getSingle()->send(Log::Error, getOrigin(), buf.str());
  }  
  catch(std::string& e) {
    Log::getSingle()->send(Log::Error, getOrigin(), "Failed to build triangulation. Reason is: "+e);
    delete sg; sg=0;
  }
  catch(...) {
    delete sg; sg = 0;
  }

  delete sg;
}

int TriangulationEditor::loadXml(const TiXmlElement* xml_node)
{
  assert(xml_node);  

  const TiXmlElement* el;
  
  el = xml_node->FirstChildElement("max_angle");
  if (el) {
    double val;
    int status = el->QueryDoubleAttribute("val", &val);  
    if (TIXML_NO_ATTRIBUTE == status) {
      max_angle = 0.1;
    }
    else max_angle = val;
  }
  
  el = xml_node->FirstChildElement("min_dist");
  if (el) {
    double val;
    int status = el->QueryDoubleAttribute("val", &val);  
    if (TIXML_NO_ATTRIBUTE == status) {
      min_dist = 0.25;
    }
    else min_dist = val;
  }

  el = xml_node->FirstChildElement("grass_dist");
  if (el) {
    double val;
    int status = el->QueryDoubleAttribute("val", &val);  
    if (TIXML_NO_ATTRIBUTE == status) {
      grass_distance = 1.0;
    }
    else grass_distance = val;
  }

  el = xml_node->FirstChildElement("grass");
  if (el)
  {
    el = el->FirstChildElement("texture");
    if (el) grass_texture.loadXml(el);
  }

  el = xml_node->FirstChildElement("side");
  if (el)
  {
    el = el->FirstChildElement("texture");
    if (el) side_texture.loadXml(el);
  }

  return 0;
}

int TriangulationEditor::saveXml(TiXmlElement* xml_node) const
{
  assert(xml_node);
  
  TiXmlElement* el;
  bool need_add = false;
  
  // max angle
  el = xml_node->FirstChildElement("max_angle");
  need_add = false;
  if (!el) {
    el = new TiXmlElement("max_angle");
    need_add = true;
  }
  el->SetDoubleAttribute("val", max_angle);
  if (need_add) {
    xml_node->InsertEndChild(*el);
    delete el;
  }
  
  // min dist
  el = xml_node->FirstChildElement("min_dist");
  need_add = false;
  if (!el) {
    el = new TiXmlElement("min_dist");
    need_add = true;
  }
  el->SetDoubleAttribute("val", min_dist);
  if (need_add) {
    xml_node->InsertEndChild(*el);
    delete el;
  }
  
  // grass distance
  el = xml_node->FirstChildElement("grass_dist");
  if (!el) {
    el = new TiXmlElement("grass_dist");
    need_add = true;
  }
  el->SetDoubleAttribute("val", grass_distance);
  if (need_add) {
    xml_node->InsertEndChild(*el);
    delete el;
  }

  // grass texture
  el = xml_node->FirstChildElement("grass");
  if (!el) {
    el = new TiXmlElement("grass");
    need_add = true;
  }
  el->Clear();
  TiXmlElement texture_node("texture");
  grass_texture.saveXml(&texture_node);
  el->InsertEndChild(texture_node);
  if (need_add) {
    xml_node->InsertEndChild(*el);
    delete el;
  }

  // side texture
  el = xml_node->FirstChildElement("side");
  if (!el) {
    el = new TiXmlElement("side");
    need_add = true;
  }
  el->Clear();
  side_texture.saveXml(&texture_node);
  el->InsertEndChild(texture_node);
  if (need_add) {
    xml_node->InsertEndChild(*el);
    delete el;
  }

  return 0;
}

}
}

void top10::tracked::TriangulationEditor::setMaxAngle( double degrees )
{
  max_angle = degrees;
}

void top10::tracked::TriangulationEditor::setMinDistance( double dist )
{
  min_dist = dist;
}

double top10::tracked::TriangulationEditor::getMaxAngle( ) const
{
  return max_angle;
}

double top10::tracked::TriangulationEditor::getMinDistance( ) const
{
  return min_dist;
}

void top10::tracked::TriangulationEditor::setGrassTexture(std::string s)
{
  grass_texture.setFilename(s);
}

void top10::tracked::TriangulationEditor::setGrassTextureScaleLat(float f)
{
  grass_texture.setScaleLat(f);
}

void top10::tracked::TriangulationEditor::setGrassTextureScaleLong(float f)
{
  grass_texture.setScaleLong(f);
}

void top10::tracked::TriangulationEditor::setGrassTextureTranslateLat(float f)
{
  grass_texture.setTranslateLat(f);
}

void top10::tracked::TriangulationEditor::setGrassTextureTranslateLong(float f)
{
  grass_texture.setTranslateLong(f);
}

void top10::tracked::TriangulationEditor::setSideTexture(std::string s)
{
  side_texture.setFilename(s);
}

void top10::tracked::TriangulationEditor::setSideTextureScaleLat(float f)
{
  side_texture.setScaleLat(f);
}

void top10::tracked::TriangulationEditor::setSideTextureScaleLong(float f)
{
  side_texture.setScaleLong(f);
}

void top10::tracked::TriangulationEditor::setSideTextureTranslateLat(float f)
{
  side_texture.setTranslateLat(f);
}

void top10::tracked::TriangulationEditor::setSideTextureTranslateLong(float f)
{
  side_texture.setTranslateLong(f);
}

void top10::tracked::TriangulationEditor::setGrassDistance(double dist)
{
  grass_distance = dist;
}

double top10::tracked::TriangulationEditor::getGrassDistance() const
{
  return grass_distance;
}
