/*
  Top 10, a racing simulator
  Copyright (C) 2003,2005  Johann Deneux
  
  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.
  
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
  
  Authors can be contacted at following electronic addresses:
  Johann Deneux: johann.deneux@it.uu.se
*/
#include "TrackFactory.hh"
#include "helpers/MeshFactory.hh"
#include "helpers/OrientMatrix.hh"
#include "util/PathFinder.hh"
#include "util/Log.hh"
#include "MainEditor.hh"

top10::tracked::TrackFactory* top10::tracked::TrackFactory::single = 0;

namespace top10 {
namespace tracked {

TrackFactory::TrackFactory()
{
}

TrackFactory::~TrackFactory()
{
}

TrackFactory* TrackFactory::getSingle()
{
  if (!single) single = new TrackFactory;
  return single;
}

top10::track::Track* TrackFactory::loadTrack(std::string filename)
{
  if (filename.size() >= 4) {
    if (filename.substr(filename.size() - 4, 4) == ".xml") return loadTrackXML(filename);
  }
  
  throw UnknownTrackType(filename);
  
  return 0;
}

top10::track::Track* TrackFactory::loadTrackXML(std::string filename)
{
  std::string pathname = top10::util::PathFinder::defaultPathFinder().find(filename);
  if (pathname.empty()) throw top10::util::Error("Could not find" + filename);
  
  MainEditor* main_ed = MainEditor::makeReadyToUse();
  assert(main_ed);
  
  main_ed->loadXml(pathname);
  
  top10::track::Track* ret = main_ed->buildTrack(filename);
  delete main_ed;
  
  return ret;
}

TrackFactory::UnknownTrackType::UnknownTrackType(std::string filename):
    top10::util::Error(filename + " uses an unknown track format. The extension must be .trk or .xml")
{
}

std::string TrackFactory::getSkyboxName(const std::string& track_name) const
{
  using top10::util::Log;

  top10::util::PathFinder finder =
    top10::util::PathFinder::defaultPathFinder();
  std::string track_path = finder.find(track_name);

  if (!track_path.empty())
    track_path = finder.getPath(track_path);
  else
  {
    Log::getSingle()->send(Log::Error, "GameManager/createTrainingSession", "Did not find track " + track_name);
    return "";
  }

  Log::getSingle()->send(Log::Debug, "GameManager/createTrainingSession", "Found track dir, will look for skybox there");
  if (finder.find(track_path + "/textures/skybox/front.jpg").empty())
  {
    Log::getSingle()->send(Log::Info, "GameManager/createTrainingSession", "Did not find track-specific skybox");
    return "";
  }
  else
  {
    Log::getSingle()->send(Log::Info, "GameManager/createTrainingSession", "Found track-specific skybox");
    return track_path + "/textures/skybox";
  }
}

}
}
