/*
  Top 10, a racing simulator
  Copyright (C) 2003,2006  Johann Deneux
  
  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.
  
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
  
  Authors can be contacted at following electronic addresses:
  Johann Deneux: johann.deneux@it.uu.se
*/
#include "TexturesEditor.hh"
#include <algorithm>

namespace top10 {
namespace tracked {

TexturesEditor::TexturesEditor():
  Drawable("textures")
{
}

void TexturesEditor::clearState()
{
  textures.clear();
}

std::string TexturesEditor::getOrigin() const
{
  return "TexturesEditor";
}

int TexturesEditor::loadXml(const TiXmlElement* node)
{
  assert(node);
  
  const TiXmlElement* child = node->FirstChildElement("texture");
  while (child)
  {
    std::string name = child->Attribute("name");
    if (name.empty())
    {
      logXmlNodeError("Missing name attribute", child, top10::util::Log::Error);
    }
    else
    {
      textures.push_back(name);
    }
    
    child = child->NextSiblingElement("texture"); 
  }
  
  return 0;
}

int TexturesEditor::saveXml(TiXmlElement* node) const
{
  assert(node);
  
  // Remove children of node
  node->Clear();
  
  // Add a child for each textures
  for (std::list<std::string>::const_iterator it = textures.begin(); it != textures.end(); ++it)
  {
    TiXmlElement entry("texture");
    entry.SetAttribute("name", *it);
    node->InsertEndChild(entry);   
  }
  
  return 0;
}

int TexturesEditor::addTexture(std::string name)
{
  std::list<std::string>::const_iterator find_it = std::find(textures.begin(), textures.end(), name);
  if (find_it != textures.end()) return -1;
  textures.push_back(name);
  return 0;
}

int TexturesEditor::removeTexture(std::string name)
{
  std::list<std::string>::iterator find_it = std::find(textures.begin(), textures.end(), name);
  if (find_it == textures.end()) return -1;
  textures.erase(find_it);
  return 0;
}

std::list<std::string> TexturesEditor::getTextures() const
{
  return textures;
}

}
}





