/*
  Top10, a racing simulator
  Copyright (C) 2000-2005  Johann Deneux
  
  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.
  
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
  
  Authors can be contacted at following electronic addresses:
  Johann Deneux: johann.deneux@it.uu.se
*/

#include "StartingAreaEditor.hh"
#include "util/strconv.hh"

using namespace top10::tracked;

StartingAreaEditor::StartingAreaEditor():
  Drawable("start_areas"),
  top_node(0),
  camera(0)
{
  top_node = new Node(this);
  addChild(top_node.getPtr());
}

std::string StartingAreaEditor::getOrigin() const
{
  return "Starting area editor";
}

int StartingAreaEditor::loadXml(const TiXmlElement* xml_node)
{
  assert(xml_node);
  if (xml_node->Value() != std::string("start_areas")) {
    logUnexpectedSection("start_areas", xml_node);
    return -1;
  }
  
  const TiXmlElement* sa_el = xml_node->FirstChildElement("start_area");
  int count = 0;
  while (sa_el) {
    ++count;
    try {
      top10::math::Vector p1, p2, dir;
      const TiXmlElement* vec = sa_el->FirstChildElement("point1");
      if (!vec) throw std::string("Missing point1");
      p1.loadXml(vec);
      
      vec = sa_el->FirstChildElement("point2");
      if (!vec) throw std::string("Missing point2");
      p2.loadXml(vec);
      
      vec = sa_el->FirstChildElement("direction");
      if (!vec) throw std::string("Missing direction");
      dir.loadXml(vec);
      
      starting_area.setDirection(dir);
      starting_area.setPoint1(p1);
      starting_area.setPoint2(p2);
    }
    catch (const std::string& err) {
      logXmlNodeError(err, sa_el);
    }
    sa_el = sa_el->NextSiblingElement("start_area");
  }
  
  if (count != 1) {
    logXmlNodeError("Current version of the track editor requires exactly one starting area, not "+top10::util::toString(count),
        xml_node);
  }
  
  return 0;
}

int StartingAreaEditor::saveXml(TiXmlElement* xml_node) const
{
  assert(xml_node);
  xml_node->SetValue("start_areas");
  xml_node->Clear();
  
  TiXmlElement sa_el("start_area");
  
  TiXmlElement p1_el("point1");
  starting_area.getPoint1().saveXml(&p1_el);
  sa_el.InsertEndChild(p1_el);
  
  TiXmlElement p2_el("point2");
  starting_area.getPoint2().saveXml(&p2_el);
  sa_el.InsertEndChild(p2_el);
  
  TiXmlElement dir_el("direction");
  starting_area.getDirection().saveXml(&dir_el);
  sa_el.InsertEndChild(dir_el);
  
  xml_node->InsertEndChild(sa_el);
  
  return 0;
}

void StartingAreaEditor::pickLeft()
{
  assert(camera);

  top10::math::Vector p = starting_area.getPoint1();
  double dist = 0.0;
  const top10::math::Plane plane( top10::math::Vector(0.0, 1.0, 0.0), p.y );
  const top10::math::Frustum f = camera->getView();
  bool b = top10::math::intersectRay(plane, f.getCenter(), f.getDirection(), p, dist);

  if (b)
  {
    starting_area.setPoint1(p);
    setDirection();
  }
}

void StartingAreaEditor::pickRight()
{
  assert(camera);

  top10::math::Vector p = starting_area.getPoint2();
  double dist = 0.0;
  const top10::math::Plane plane( top10::math::Vector(0.0, 1.0, 0.0), p.y );
  const top10::math::Frustum f = camera->getView();
  bool b = top10::math::intersectRay(plane, f.getCenter(), f.getDirection(), p, dist);

  if (b)
  {
    starting_area.setPoint2(p);
    setDirection();
  }
}

void StartingAreaEditor::swap()
{
  top10::math::Vector tmp = starting_area.getPoint1();
  starting_area.setPoint1(starting_area.getPoint2());
  starting_area.setPoint2(tmp);
  setDirection();
}

void StartingAreaEditor::setDirection()
{
  top10::math::Vector dir = (starting_area.getPoint1() - starting_area.getPoint2())^top10::math::Vector(0,1,0);
  double s = dir.size();
  if (s < SMALL_VECTOR) throw std::string("Left and right points cannot be aligned vertically");
  dir /= s;
  starting_area.setDirection(dir);
}

void StartingAreaEditor::Node::renderGL(const top10::graphX::RenderingFeatures&,
  const top10::graphX::RenderState&,
  const top10::graphX::CameraNode&) const
{
  using top10::math::Vector;
  using top10::racing::StartingArea;

  Vector p = ed->starting_area.getPoint1();
 
  glBegin(GL_LINE_STRIP);
  glVertex3f(p.x, p.y, p.z);
  p -= 5*ed->starting_area.getDirection();
  glVertex3f(p.x, p.y, p.z);
  p = ed->starting_area.getPoint2();
  p -= 5*ed->starting_area.getDirection();
  glVertex3f(p.x, p.y, p.z);
  p = ed->starting_area.getPoint2();
  glVertex3f(p.x, p.y, p.z);
  glEnd();
}

void StartingAreaEditor::clearState( )
{
  starting_area = top10::racing::StartingArea();
}
