/*
  Top 10, a racing simulator
  Copyright (C) 2003,2005  Johann Deneux
  
  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.
  
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
  
  Authors can be contacted at following electronic addresses:
  Johann Deneux: johann.deneux@it.uu.se
*/

#ifndef TOP10_TRACKEDSECTIONSEDITOR_HH
#define TOP10_TRACKEDSECTIONSEDITOR_HH

#include "Drawable.hh"
#include "PathEditor.hh"
#include "track/SectionGraph.hh"
#include "track/Waypoint.hh"
#include "track/TextureSpec.hh"
#include <tinyxml.h>
#include "graphX/MaterialNode.hh"

namespace top10 {
namespace tracked {

//! Maintains connections between waypoints
/*! The waypoints are managed by the PathEditor. */
class SectionsEditor: public Drawable {
public:
  //! A point controlling one end of a section.
  struct ControlPoint {
    //! Id of the waypoint associated to this point.
    int waypoint_id;
    //! Size multiplier of the tangent when used as the 1st point of a section
    float tg_size_1;
    //! Size multiplier of the tangent when used as the 2nd point of a section
    float tg_size_2;
    //! Distance of the left border of the road from the "center"
    float left_road_width;
    //! Distance of the right border of the road from the "center"
    float right_road_width;
    //! Which tangent to use (current valid values: 0 (primary) or 1 (secondary tangent)
    int tg_use_n;
  };
  
  struct Path: public top10::track::TextureSpec {
    //! All the EndPoints composing this path
    std::vector<ControlPoint> pts;
    //! Name of this path
    std::string name;
    //! Indicates if the last EndPoint is connected to the first
    bool loops;
    
    //! Indicate the ControlPoint on which the editor points
    /*! This variable does not describe a path, it is only useful for the editor */
    int current_control;
  };
  
public:
  SectionsEditor();
  
  //! Erase all data
  void clearState();
  
  // Implementation of Drawable
  int loadXml(const TiXmlElement* xml_node);
  int saveXml(TiXmlElement* xml_node) const;
  std::string getOrigin() const;
  
  void updateView();

  inline void setWaypointsEditor(PathEditor* p) { waypoints_ed = p; }
  inline PathEditor* getWaypointsEditor() { return waypoints_ed; }
  inline const PathEditor* getWaypointsEditor() const { return waypoints_ed; }

  /*
    Methods intended to be called by dialogs to change the state of this editor.
  */
  //! Set the current path according to its name
  void gotoPath(std::string path_name);
  
  //! Move to the next control point
  void gotoNextControl();
  //! Move to the previous control point
  void gotoPrevControl();
  //! Move directly to the idx-th control point
  void gotoControl(int idx);
  //! Make room for a new control point
  /*! The current waypoint in the waypoint editor is used for this new control point */
  void addControl();
  //! Remove the current control point
  void removeControl();
  //! Set the waypoint by its name
  void setWaypoint(const std::string&);
  //! Set the incoming tangent factor
  void setTgFactorIn(float);
  //! Set the outgoing tangent factor
  void setTgFactorOut(float);
  //! Set the left width of the road
  void setLRoadWidth(float);
  //! Set the right width of the road
  void setRRoadWidth(float);
  //! Set which tangent to use (0-> primary, 1->secondary)
  void setTgUse(int);
  
  //! Create a new path
  void addPath();
  //! Remove the current path and all its sections
  void removePath();
  //! Rename the current path
  void setPathName(std::string);
  //! Set the texture name
  void setTexture(std::string);
  //! Set the lateral texture scaling factor
  void setTextureScaleLat(float);
  //! Set the forward texture scaling factor
  void setTextureScaleLong(float);
  //! Set the lateral texture translation factor
  void setTextureTranslateLat(float);
  //! Set the forward texture translation factor
  void setTextureTranslateLong(float);
  //! Set/Unset the flag telling if the current path loops from the last control point to the first.
  void setLoops(bool);

  //! Change tangents of all waypoints in the path to conveniently create smooth curves
  void smoothTangents();
  
  /*
    Query methods intended to be called by dialogs to populate their fields.
  */
  //! Get the position of the current control point, or -1
  int getCurrentControlIndex() const;
  //! Get the position of the last control point in the current path, or -1
  int getLastControlIndex() const;
  //! Get the left width of the current control, or 0.0
  float getLRoadWidth() const;
  //! Get the right width of the current control, or 0.0
  float getRRoadWidth() const;
  //! Get the tangent factor of the incoming section, or 0.0
  float getTgFactorIn() const;
  //! Get the tangent factor of the outgoing section, or 0.0
  float getTgFactorOut() const;
  //! Get the name of the waypoint, or the empty string
  std::string getWaypointName() const;
  //! Return the number of the tangent to use
  int getTgUse() const;

  //! Get the name of the current path, or the empty string
  std::string getPathName() const;
  //! Get the names of all paths
  std::list<std::string> getPathNames() const;
  //! Return the address of the TextureSpec of the current section, or 0
  const top10::track::TextureSpec* getTextureSpec() const;
  //! Get the name of the texture
  std::string getTexture() const;
  //! Get the lateral texture scaling factor
  float getTextureScaleLat() const;
  //! Get the forward texture scaling factor
  float getTextureScaleLong() const;
  //! Get the lateral texture translation factor
  float getTextureTranslateLat() const;
  //! Get the forward texture translation factor
  float getTextureTranslateLong() const;
  //! Get the flag telling if this path loops.
  bool getLoops() const;
  //! Get the length of the current path, or 0.
  double getCurrentPathLength() const;
  
  /*
    Methods used to build a track.
  */
  top10::track::SectionGraph* makeSectionGraph();
  //! For debugging
#if 0
  top10::track::TrackTriangulation* makeTriangulation(bool invert);
#endif

private:
  std::list<Path> paths;
  std::list<Path>::iterator current_path;

private:
  //! Thrown by getWayPoint(Control)
  class NoSuchWaypoint {};
  
  //! Base class used to do something for each section in all paths
  /*! What is actually done is specified by subclassing. */
  class Computable {
  public:
    Computable(SectionsEditor* ed): ed(ed) {}
    void compute();
    virtual void handleSection(const top10::track::Section&) =0;
    virtual void finish() =0;
    virtual ~Computable() {};
  protected:
    SectionsEditor* ed;
  };
  
  //! Compute the section graph
  class SectionGraphMaker: public Computable
  {
  public:
    SectionGraphMaker(SectionsEditor* ed);
    void handleSection(const top10::track::Section&);
    void finish();
    inline top10::track::SectionGraph* getSectionGraph() { return sg; }
  private:
    top10::track::SectionGraph* sg;
  };
  
  //! Render the outline of the road
  /*! What is rendered is the vectorization of each section, not the outline of the whole track */
  class SimpleSectionsNode: public Computable, public top10::graphX::LeafNode {
  public:
    SimpleSectionsNode(SectionsEditor* ed): Computable(ed), max_angle(0.3), min_dist(0.5) {}
    void renderGL(const top10::graphX::RenderingFeatures&, const top10::graphX::RenderState&, const top10::graphX::CameraNode&) const;
  private:
    double max_angle;
    double min_dist;
    
    std::list< std::vector< top10::math::Vector > > lefts;
    std::list< std::vector< top10::math::Vector > > rights;
    
    void handleSection(const top10::track::Section&);
    void finish();
    void drawPoint(top10::math::Vector v) const;
  };
  
  //! Show the current control point
  class CursorNode: public top10::graphX::LeafNode {
  public:
    CursorNode(SectionsEditor* ed): ed(ed), max_angle(0.3), min_dist(0.5) {}
    void renderGL(const top10::graphX::RenderingFeatures&, const top10::graphX::RenderState&, const top10::graphX::CameraNode&) const;
  private:
    SectionsEditor* ed;
    double max_angle;
    double min_dist;
    
    void drawSectionPart(int idx1, int idx2, float t1, float t2) const;
  };
    
  PathEditor* waypoints_ed;
  top10::util::Ref<CursorNode> cursor_node;
  top10::util::Ref<SimpleSectionsNode> simple_node;
  top10::util::Ref<top10::graphX::MaterialNode> color_node;
  
  unsigned char color_r;
  unsigned char color_g;
  unsigned char color_b;

  //! Compute the index of the next point in the current path
  int nextControl(int) const;
  //! Compute the index of the previous point in the current path
  int prevControl(int) const;
  
  //! Load one path
  Path loadPathXml(const TiXmlElement* path_node);
  //! Save one path
  TiXmlElement savePathXml(const Path& path) const;
  
  //! Check the correctness of various pointers/indices
  bool isCorrect() const;
  
  //! Find a path by its name
  std::list<Path>::iterator findPathNamed(std::string path_name);
  
  //! Return the control point in the current path at a given position
  /*! Aborts if the point does not exist. */
  ControlPoint getControl(int idx) const;
  
  //! Get the waypoint of a control point.
  /*! \throw NoSuchWaypoint if the waypoint is not found in the waypoints editor */
  top10::track::Waypoint getWaypoint(ControlPoint) const;
  
  friend class SectionsEditor::Computable;
  friend class SectionsEditor::SectionGraphMaker;
  friend class SectionsEditor::CursorNode;
  friend class SectionsEditor::SimpleSectionsNode;
};

}
}

#endif
