/*
  Top 10, a racing simulator
  Copyright (C) 2003,2005  Johann Deneux
  
  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.
  
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
  
  Authors can be contacted at following electronic addresses:
  Johann Deneux: johann.deneux@it.uu.se
*/
#include "SectionsEditor.hh"
#include "util/Log.hh"
#include "util/strconv.hh"
#include "util/findpred.hh"
#include "math/Hermite.hh"
#include "track/Section.hh"

#include <sstream>

using top10::util::Log;

namespace top10 {
namespace tracked {

SectionsEditor::SectionsEditor():
  Drawable("paths"),
  current_path(paths.end()), waypoints_ed(0),
  color_r(255), color_g(255), color_b(0)
{
  color_node = new top10::graphX::MaterialNode;
  color_node->r = color_r;
  color_node->g = color_g;
  color_node->b = color_b;
  
  simple_node = new SimpleSectionsNode(this);
  simple_node->compute();
  color_node->addChild(simple_node.getPtr());
  
  cursor_node = new CursorNode(this);
  
  addChild(cursor_node.getPtr());
  addChild(color_node.getPtr());
}

void SectionsEditor::clearState()
{
  paths.clear();
  current_path = paths.end();
}

void SectionsEditor::updateView()
{
  if (!isVisible()) return;
  
  color_node->removeChild(simple_node.getPtr());
  simple_node = new SimpleSectionsNode(this);
  
  try {
    simple_node->compute();
    color_node->addChild(simple_node.getPtr());
  }
  catch(top10::util::Error& e) {
    Log::getSingle()->send(Log::Error, getOrigin(), e);
  }
  catch(...) {
    Log::getSingle()->send(Log::Error, getOrigin(), "Unknown error");
  }
}

std::string SectionsEditor::getOrigin() const
{
  return "SectionsEditor";
}

int SectionsEditor::loadXml(const TiXmlElement* xml_node)
{
  assert(xml_node);
  
  if (xml_node->Value() != std::string("paths")) {
    Log::getSingle()->send(Log::Error, getOrigin(), std::string("Expected paths, got ")+xml_node->Value()+
      " on line "+top10::util::toString(xml_node->Row())+
      " column "+top10::util::toString(xml_node->Column()));
    return -1;
  }
  
  std::list<Path> tmp_paths;
  
  // Iterate through the children
  const TiXmlElement* wp_node = xml_node->FirstChildElement("path");
  while (wp_node) {
    try {
      Path new_one = loadPathXml(wp_node);
      tmp_paths.push_back(new_one);
    }
    catch (const std::string& err) {
      Log::getSingle()->send(Log::Warning, getOrigin()+"/XmlParser", err +
          " line "+top10::util::toString(wp_node->Row())+
          " column "+top10::util::toString(wp_node->Column()));
    }
    
    wp_node = wp_node->NextSiblingElement("path");
  }
  
  // Swap our paths with the newly loaded ones
  paths.swap(tmp_paths);
  // Point on the first one.
  current_path = paths.begin();
  
  updateView();
  
  return 0;
}

SectionsEditor::Path SectionsEditor::loadPathXml(const TiXmlElement* path_node)
{
  assert(path_node);
  
  // The variable we'll return
  Path path;
  path.current_control = -1;
  
  // Looping enabled?
  int loops=0;
  path_node->Attribute("loops", &loops);
  path.loops = loops!=0;
  const char* name = path_node->Attribute("name");
  if (name)
    path.name = name;
  else
    Log::getSingle()->send(Log::Error, getOrigin(), "Missing name of path");

  // Texture
  const TiXmlElement* texture_node = path_node->FirstChildElement("texture");
  if (texture_node)
    path.loadXml(texture_node);

  // Iterate over list of endpoints
  const TiXmlElement* ep_node = path_node->FirstChildElement("endpoint");
  while (ep_node) {
    try {
      ControlPoint new_one;
      int status;
      status = ep_node->QueryIntAttribute("id", &new_one.waypoint_id);
      if (status == TIXML_WRONG_TYPE) throw std::string("Wrong type, expected int");
      else if (status == TIXML_NO_ATTRIBUTE) throw std::string("Missing attribute: id");
      
      status = ep_node->QueryFloatAttribute("width", &new_one.left_road_width);
      if (status == TIXML_NO_ATTRIBUTE) {
        new_one.left_road_width = 3.0;
        ep_node->QueryFloatAttribute("left_road_width", &new_one.left_road_width);
      
        new_one.right_road_width = 3.0;
        ep_node->QueryFloatAttribute("right_road_width", &new_one.right_road_width);
      }
      else {
        new_one.right_road_width = new_one.left_road_width = new_one.left_road_width/2.0;
      }
      
      new_one.tg_size_1 = 1.0;
      ep_node->QueryFloatAttribute("factor1", &new_one.tg_size_1);
      
      new_one.tg_size_2 = 1.0;
      ep_node->QueryFloatAttribute("factor2", &new_one.tg_size_2);

      new_one.tg_use_n = 0;
      ep_node->QueryIntAttribute("tg_n", &new_one.tg_use_n);

      path.pts.push_back(new_one);
    }
    catch (const std::string& err) {
      Log::getSingle()->send(Log::Error, getOrigin()+"/XmlParser", err +
          " line "+top10::util::toString(ep_node->Row())+
          " column "+top10::util::toString(ep_node->Column()));
    }
    
    ep_node = ep_node->NextSiblingElement("endpoint"); 
  }
  
  return path;
}

int SectionsEditor::saveXml(TiXmlElement* xml_node) const
{
  assert(xml_node);
  
  // Remove all children
  xml_node->Clear();
  
  // Add a child for each path
  for (std::list<Path>::const_iterator path_it = paths.begin(); path_it != paths.end(); ++path_it) {
    xml_node->InsertEndChild(savePathXml(*path_it));
  }
  
  return 0;
}

TiXmlElement SectionsEditor::savePathXml(const Path& path) const
{
  TiXmlElement new_one("path");
  
  new_one.SetAttribute("loops", path.loops?1:0);
  new_one.SetAttribute("name", path.name);

  TiXmlElement texture_node("texture");
  path.saveXml(&texture_node);
  new_one.InsertEndChild(texture_node);

  // For each endpoint
  for (std::vector<ControlPoint>::const_iterator pt_it = path.pts.begin(); pt_it != path.pts.end(); ++pt_it) {
    TiXmlElement pt_element("endpoint");
    pt_element.SetAttribute("id", pt_it->waypoint_id);
    pt_element.SetDoubleAttribute("left_road_width", pt_it->left_road_width);
    pt_element.SetDoubleAttribute("right_road_width", pt_it->right_road_width);
    pt_element.SetDoubleAttribute("factor1", pt_it->tg_size_1);
    pt_element.SetDoubleAttribute("factor2", pt_it->tg_size_2);
    pt_element.SetAttribute("tg_n", pt_it->tg_use_n);
    new_one.InsertEndChild(pt_element);
  }
  
  return new_one;
}

bool SectionsEditor::isCorrect() const
{
  if (current_path != paths.end()) {
    // current_control in range
    if (current_path->current_control < -1) {
      Log::getSingle()->send(Log::Critical, getOrigin(), "current_path->current_control < -1");
      return false;
    }
    if (current_path->current_control >= (int)current_path->pts.size()) {
      std::ostringstream buf;
      buf<<"current_path->current_control ("<<current_path->current_control<<") >= (int)current_path->pts.size() ("<<(int)current_path->pts.size()<<")";
      Log::getSingle()->send(Log::Critical, getOrigin(), buf.str());
      return false;
    }
  }
  
  return true;
}

void SectionsEditor::gotoPath(std::string path_name)
{
  assert(isCorrect());
  std::list<Path>::iterator it = findPathNamed(path_name);
  if (it != paths.end()) current_path = it;
  else {
    Log::getSingle()->send(Log::Warning, getOrigin(), std::string("Path named "+path_name+" not found"));
  }
}

std::list<SectionsEditor::Path>::iterator SectionsEditor::findPathNamed(std::string path_name)
{
  assert(isCorrect());
  top10::util::FindNamed finder(path_name);
  return std::find_if(paths.begin(), paths.end(), finder);
}

int SectionsEditor::nextControl(int section) const
{
  assert(isCorrect());
  
  if (current_path != paths.end()) {
    if (current_path->pts.empty()) return -1;
    else if (section == (int)current_path->pts.size() -1) return 0;
    else return section+1;
  }
  
  return -1;
}

int SectionsEditor::prevControl(int section) const
{
  assert(isCorrect());
  
  if (current_path != paths.end()) {
    if (current_path->pts.empty()) return -1;
    else if (section <= 0) return current_path->pts.size()-1;
    else return section-1;
  }
  
  return -1;
}

void SectionsEditor::gotoControl(int idx)
{
  assert(isCorrect());

  if (current_path != paths.end()) {
    if (idx >= 0 && idx < (int)current_path->pts.size())
    {
      current_path->current_control = idx;
    }
  }
  else Log::getSingle()->send(Log::Error, getOrigin(), std::string("Please select a path first."));
}

void SectionsEditor::gotoNextControl()
{
  assert(isCorrect());
  
  if (current_path != paths.end()) {
    current_path->current_control = nextControl(current_path->current_control);
  }
  else Log::getSingle()->send(Log::Error, getOrigin(), std::string("Please select a path first."));
}

void SectionsEditor::gotoPrevControl()
{
  assert(isCorrect());
  
  if (current_path != paths.end()) {
    current_path->current_control = prevControl(current_path->current_control);
  }
  else Log::getSingle()->send(Log::Error, getOrigin(), std::string("Please select a path first."));
}

void SectionsEditor::setLoops(bool b)
{
  assert(isCorrect());
  
  if (current_path != paths.end()) {
    current_path->loops = b;
  }
  else Log::getSingle()->send(Log::Error, getOrigin(), std::string("Please select a path first."));
  
  updateView();
}

void SectionsEditor::smoothTangents()
{
  assert(isCorrect());
  if (current_path == paths.end()) return;
  if (current_path->pts.size() < 3)
  {
    Log::getSingle()->send(Log::Error, getOrigin(), std::string("Too few points in this path"));
    return;
  }

  for (std::vector<ControlPoint>::const_iterator it = current_path->pts.begin();
    it != current_path->pts.end();
    ++it)
  {
    int prev_idx, curr_idx, next_idx;
    if (it == current_path->pts.begin())
    {
      if (current_path->loops)
      {
	prev_idx = waypoints_ed->findById(current_path->pts.back().waypoint_id);
	curr_idx = waypoints_ed->findById(it->waypoint_id);
	next_idx = waypoints_ed->findById((it+1)->waypoint_id);    
      }
      else continue;
    }
    else if (it+1 == current_path->pts.end())
    {
      if (current_path->loops)
      {
	prev_idx = waypoints_ed->findById((it-1)->waypoint_id);
	curr_idx = waypoints_ed->findById(it->waypoint_id);
	next_idx = waypoints_ed->findById(current_path->pts.front().waypoint_id);    
      }
      else continue;
    }
    else
    {
      prev_idx = waypoints_ed->findById((it-1)->waypoint_id);
      curr_idx = waypoints_ed->findById(it->waypoint_id);
      next_idx = waypoints_ed->findById((it+1)->waypoint_id);    
    }

    top10::math::Vector prev_diff, next_diff;
    next_diff = waypoints_ed->waypoints.at(next_idx).pos - waypoints_ed->waypoints.at(curr_idx).pos;
    prev_diff = waypoints_ed->waypoints.at(curr_idx).pos - waypoints_ed->waypoints.at(prev_idx).pos;
    double next_sz = next_diff.size();
    double prev_sz = prev_diff.size();
    if (next_sz >= SMALL_VECTOR && prev_sz >= SMALL_VECTOR)
    {
      next_diff /= next_sz;
      prev_diff /= prev_sz;
      top10::math::Vector avg_diff = prev_diff + next_diff;
      double avg_sz = avg_diff.size();
      if (avg_sz >= SMALL_VECTOR)
      {
	avg_diff /= avg_sz;
	switch(it->tg_use_n)
	{
	case 0:
	  waypoints_ed->setCurrentTangent(avg_diff, it->waypoint_id);
	  break;
	case 1:
	  waypoints_ed->setCurrentTangent2(avg_diff, it->waypoint_id);
	  break;
	default:
	  Log::getSingle()->send(Log::Error, getOrigin(), "Invalid tangent number "+top10::util::toString(it->tg_use_n)+", should be 0 or 1");
	}
      }
    }
  }

  updateView();
}

bool SectionsEditor::getLoops() const
{
  assert(isCorrect());
  
  if (current_path != paths.end()) return current_path->loops;
  return false;
}

double SectionsEditor::getCurrentPathLength() const
{
  assert(isCorrect());

  if (current_path == paths.end() || current_path->pts.size() <= 1) return 0.0;

  double length = 0.0;
  std::vector<ControlPoint>::const_iterator pt_it = current_path->pts.begin();
  std::vector<ControlPoint>::const_iterator pt2_it = pt_it;

  for (; pt_it != current_path->pts.end(); ++pt_it, ++pt2_it)
  {
    // Last iteration only executed if looping is enabled.
    if (pt2_it == current_path->pts.end()) {
      if (current_path->loops) pt2_it = current_path->pts.begin();
      else break;
    }

    // Build the curve going through the endpoints
    top10::math::Hermite curve;
    int idx1 = getWaypointsEditor()->findById(pt_it->waypoint_id);
    int idx2 = getWaypointsEditor()->findById(pt2_it->waypoint_id);
    if (idx1 != -1 && idx2 != -1)
    {
      curve.setEnd1(getWaypointsEditor()->waypoints.at(idx1).pos);
      curve.setEnd2(getWaypointsEditor()->waypoints.at(idx2).pos);
      curve.setTangent1(getWaypointsEditor()->waypoints.at(idx1).tg * pt_it->tg_size_1);
      curve.setTangent2(getWaypointsEditor()->waypoints.at(idx2).tg * pt2_it->tg_size_2);
      length += curve.getLength(0.0, 1.0);
    }
  }

  return length;
}

int SectionsEditor::getCurrentControlIndex() const
{
  assert(isCorrect());
  
  if (current_path != paths.end()) return current_path->current_control;
  return -1;
}

int SectionsEditor::getLastControlIndex() const
{
  assert(isCorrect());
  
  if (current_path != paths.end()) return current_path->pts.size() -1;
  return -1;
}

void SectionsEditor::addControl()
{
  assert(isCorrect());
  assert(waypoints_ed);
  
  if (!waypoints_ed->waypoints.empty()) {
    if (current_path != paths.end()) {
      // Prepare a new control point
      ControlPoint new_one;
      new_one.left_road_width = new_one.right_road_width = 4.0;
      new_one.tg_size_1 = 10.0;
      new_one.tg_size_2 = 10.0;
      
      // Set the id of the waypoint
      if (waypoints_ed->getCurrentIndex() != -1)
        new_one.waypoint_id = waypoints_ed->getCurrentWayPoint().id;
      else new_one.waypoint_id = waypoints_ed->waypoints.at(0).id;
      
      // Insert after the current control
      if (current_path->current_control != -1)
        current_path->pts.insert(current_path->pts.begin()+(current_path->current_control+1), new_one);
      // Insert as the first control
      else current_path->pts.push_back(new_one);
      
      // Move to the newly added control
      gotoNextControl();
      
      updateView();
    }
    else Log::getSingle()->send(Log::Error, getOrigin(), std::string("Please select a path first."));
  }
  else Log::getSingle()->send(Log::Error, getOrigin(), std::string("Please create a waypoint first."));
  
  Log::getSingle()->send(Log::Info, getOrigin(), "Created new control point.");
}

void SectionsEditor::removeControl()
{
  assert(isCorrect());
  
  if (current_path != paths.end()) {
    if (!current_path->pts.empty()) {
      if (getCurrentControlIndex() > -1)
        current_path->pts.erase(current_path->pts.begin() + getCurrentControlIndex());
      if (current_path->current_control == current_path->pts.size())
	current_path->current_control--;
      else
	gotoPrevControl();
      
      updateView();
    }
    else Log::getSingle()->send(Log::Error, getOrigin(), std::string("No control point to remove in ") + getPathName());
  }
  else Log::getSingle()->send(Log::Error, getOrigin(), std::string("Please select a path first."));
}

void SectionsEditor::addPath()
{
  Path new_path;
  new_path.current_control = -1;
  new_path.loops = false;
  int i=1;
  do {
    std::ostringstream buf;
    buf<<"Unnamed_"<<i;
    ++i;
    new_path.name = buf.str();
  } while (findPathNamed(new_path.name) != paths.end());
  paths.push_front(new_path);
  current_path = paths.begin();
  
  Log::getSingle()->send(Log::Info, getOrigin(), "Created " + new_path.name);
}

void SectionsEditor::removePath()
{
  if (current_path != paths.end()) {
    paths.erase(current_path);
    current_path = paths.begin();
    
    updateView();
  }
}

void SectionsEditor::setPathName(std::string name)
{
  if (current_path != paths.end()) {
    current_path->name = name;
  }
}

std::string SectionsEditor::getPathName() const
{
  if (current_path != paths.end()) {
    return current_path->name;
  }
  return "";
}

std::list<std::string> SectionsEditor::getPathNames() const
{
  std::list<std::string> ret;
  for (std::list<Path>::const_iterator it = paths.begin(); it != paths.end(); ++it)
    ret.push_back(it->name);
  return ret;
}

std::string SectionsEditor::getWaypointName() const
{
  if (getCurrentControlIndex() == -1) return "";
  int id = current_path->pts.at(current_path->current_control).waypoint_id;
  int idx = waypoints_ed->findById(id);
  if (idx == -1) return "";
  return waypoints_ed->waypoints.at(idx).name;
}

float SectionsEditor::getTgFactorIn() const
{
  if (getCurrentControlIndex() == -1) return 0.0;
  return current_path->pts.at(current_path->current_control).tg_size_1;
}

float SectionsEditor::getTgFactorOut() const
{
  if (getCurrentControlIndex() == -1) return 0.0;
  return current_path->pts.at(current_path->current_control).tg_size_2;
}

float SectionsEditor::getLRoadWidth() const
{
  if (getCurrentControlIndex() == -1) return 0.0;
  return current_path->pts.at(current_path->current_control).left_road_width;
}

float SectionsEditor::getRRoadWidth() const
{
  if (getCurrentControlIndex() == -1) return 0.0;
  return current_path->pts.at(current_path->current_control).right_road_width;
}

const top10::track::TextureSpec* SectionsEditor::getTextureSpec() const
{
  if (current_path == paths.end()) return 0;
  return &*current_path;
}

std::string SectionsEditor::getTexture() const
{
  if (current_path == paths.end()) return "";
  return current_path->getFilename();
}

float SectionsEditor::getTextureScaleLat() const
{
  if (current_path == paths.end()) return 0.0;
  return current_path->getScaleLat();
}

float SectionsEditor::getTextureScaleLong() const
{
  if (current_path == paths.end()) return 0.0;
  return current_path->getScaleLong();
}

float SectionsEditor::getTextureTranslateLat() const
{
  if (current_path == paths.end()) return 0.0;
  return current_path->getTranslateLat();
}

float SectionsEditor::getTextureTranslateLong() const
{
  if (current_path == paths.end()) return 0.0;
  return current_path->getTranslateLong();
}

int SectionsEditor::getTgUse() const
{
  if (getCurrentControlIndex() == -1) return 0;
  return current_path->pts.at(current_path->current_control).tg_use_n;
}

void SectionsEditor::setWaypoint(const std::string& wp_name)
{
  assert(waypoints_ed);
  if (getCurrentControlIndex() == -1) return;
  int idx = waypoints_ed->findNamed(wp_name);
  if (idx == -1) return;
  current_path->pts.at(current_path->current_control).waypoint_id = waypoints_ed->waypoints.at(idx).id;
  
  updateView();
}

void SectionsEditor::setTgFactorIn(float x)
{
  if (getCurrentControlIndex() == -1) return;
  current_path->pts.at(current_path->current_control).tg_size_1 = x;
  
  updateView();
}

void SectionsEditor::setTgFactorOut(float x)
{
  if (getCurrentControlIndex() == -1) return;
  current_path->pts.at(current_path->current_control).tg_size_2 = x;
  
  updateView();
}

void SectionsEditor::setTgUse(int n)
{
  if (getCurrentControlIndex() == -1) return;
  current_path->pts.at(current_path->current_control).tg_use_n = n;

  updateView();
}

void SectionsEditor::setLRoadWidth(float x)
{
  if (getCurrentControlIndex() == -1) return;
  current_path->pts.at(current_path->current_control).left_road_width = x;
  
  updateView();
}

void SectionsEditor::setRRoadWidth(float x)
{
  if (getCurrentControlIndex() == -1) return;
  current_path->pts.at(current_path->current_control).right_road_width = x;
  
  updateView();
}

void SectionsEditor::setTexture(std::string name)
{
  if (current_path == paths.end()) return;
  current_path->setFilename(name);
}

void SectionsEditor::setTextureScaleLat(float x)
{
  if (current_path == paths.end()) return;
  current_path->setScaleLat(x);
}

void SectionsEditor::setTextureScaleLong(float x)
{
  if (current_path == paths.end()) return;
  current_path->setScaleLong(x);
}

void SectionsEditor::setTextureTranslateLat(float x)
{
  if (current_path == paths.end()) return;
  current_path->setTranslateLat(x);
}

void SectionsEditor::setTextureTranslateLong(float x)
{
  if (current_path == paths.end()) return;
  current_path->setTranslateLong(x);
}

SectionsEditor::ControlPoint SectionsEditor::getControl(int idx) const
{
  assert(current_path != paths.end());
  assert(idx >= 0);
  assert(idx < current_path->pts.size());
  
  return current_path->pts.at(idx);
}

top10::track::Waypoint SectionsEditor::getWaypoint(ControlPoint ctrl) const
{
  assert(waypoints_ed);
  int idx = waypoints_ed->findById(ctrl.waypoint_id);
  if (idx == -1) throw NoSuchWaypoint();
  top10::track::Waypoint ret;
  ret.setPos(waypoints_ed->waypoints.at(idx).pos);
  ret.setBankAngle(waypoints_ed->waypoints.at(idx).bank_angle);
  ret.setTangent(waypoints_ed->waypoints.at(idx).tg);
  return ret;
}

top10::track::SectionGraph* SectionsEditor::makeSectionGraph()
{
  SectionGraphMaker maker(this);
  maker.compute();
  return maker.getSectionGraph();
}

#if 0
top10::track::TrackTriangulation* SectionsEditor::makeTriangulation(bool invert)
{
  top10::track::SectionGraph* sg = makeSectionGraph();
  sg->finishCreate();
  std::list< std::vector< top10::track::TexturedVertex > > outlines = sg->getOutlines();
  
  top10::track::TexturedVertexAccessor acc;
  top10::track::TrackTriangulation* ret = new top10::track::TrackTriangulation(-1e4, -1e4, 1e4, 1e4, acc);
  
  for (std::list< std::vector< top10::track::TexturedVertex > >::const_iterator it = outlines.begin();
       it != outlines.end();
       ++it)
  {
    if (invert)
      ret->addConstraintsInv(*it);
    else
      ret->addConstraints(*it);
  }
  
    // Add all waypoints, so that lonely points also appear in the triangulation
  if (getWaypointsEditor()) {
    for (std::vector<PathEditor::WayPoint>::const_iterator it = getWaypointsEditor()->waypoints.begin();
         it != getWaypointsEditor()->waypoints.end();
         ++it)
    {
      top10::track::TexturedVertex tv;
      tv.pos = it->pos;
      
      // Prepare to make an AreaCoord suitable for grass
      top10::track::PosCoordMaker ac_maker(top10::track::GrassAreaId);
      top10::track::OutlineVertex ov;
      ov.center = it->pos;
      ov.normal = top10::math::Vector(0.0, 0.0, 0.0);
      ov.t = 0.0;
      ov.u = 0.0;
      ov.v = 0.0;
      
      // Make the AreaCoord and add it.
      tv.coords.push_back(ac_maker.make(ov));
      
      // Add the new vertex
      ret->addVertex(tv);
    }
  }
  
  return ret;
}

#endif

void SectionsEditor::Computable::compute()
{
  for (std::list<Path>::const_iterator path_it = ed->paths.begin(); path_it != ed->paths.end(); ++path_it)
  {
    double length = 0.0;
    std::vector<SectionsEditor::ControlPoint>::const_iterator pt_it = path_it->pts.begin();
    std::vector<SectionsEditor::ControlPoint>::const_iterator pt2_it = pt_it;

    // No points in the current path
    if (pt2_it == path_it->pts.end()) continue;
    ++pt2_it;
    // Only one point in the path
    if (pt2_it == path_it->pts.end()) continue;

    for (; pt_it != path_it->pts.end(); ++pt_it, ++pt2_it) {
      // Last iteration only executed if looping is enabled.
      if (pt2_it == path_it->pts.end()) {
        if (path_it->loops) pt2_it = path_it->pts.begin();
	else break;
      }

      // Build the curve going through the endpoints
      top10::math::Hermite curve;
      int idx1 = ed->getWaypointsEditor()->findById(pt_it->waypoint_id);
      int idx2 = ed->getWaypointsEditor()->findById(pt2_it->waypoint_id);
      if (idx1 != -1 && idx2 != -1) {
        curve.setEnd1(ed->getWaypointsEditor()->waypoints.at(idx1).pos);
        curve.setEnd2(ed->getWaypointsEditor()->waypoints.at(idx2).pos);
        curve.setTangent1(ed->getWaypointsEditor()->waypoints.at(idx1).tg * pt_it->tg_size_1);
        curve.setTangent2(ed->getWaypointsEditor()->waypoints.at(idx2).tg * pt2_it->tg_size_2);

        // Create the section
        top10::track::Section track_section;
        track_section.setCurve(curve);
        track_section.setLeftDist1(-pt_it->left_road_width);
        track_section.setRightDist1(pt_it->right_road_width);
        track_section.setBankAngle1(M_PI*ed->getWaypointsEditor()->waypoints.at(idx1).bank_angle/180.0);
        track_section.setLeftDist2(-pt2_it->left_road_width);
        track_section.setRightDist2(pt2_it->right_road_width);
        track_section.setBankAngle2(M_PI*ed->getWaypointsEditor()->waypoints.at(idx2).bank_angle/180.0);
	track_section.setTextureSpec(ed->getTextureSpec());
	track_section.setAbsStart(length);
	length += curve.getLength(0.0, 1.0);
        handleSection(track_section);
      }
    }
  }

  finish();
}

SectionsEditor::SectionGraphMaker::SectionGraphMaker(SectionsEditor* ed): Computable(ed)
{
  double x_min=1e6, z_min=1e6, x_max=-1e6, z_max=-1e6;
  ed->getWaypointsEditor()->getMinMax(&x_min, &z_min, &x_max, &z_max);
  sg = new top10::track::SectionGraph(x_min-500.0, z_min-500.0, x_max+500.0, z_max+500.0);
}

void SectionsEditor::SectionGraphMaker::handleSection(const top10::track::Section& track_section)
{
  // Add it to the track graph
  try {
    sg->addSection(track_section);
  }
  catch (top10::track::SectionGraph::SameExists& e) {
    // Don't do anything
  }
}

void SectionsEditor::SectionGraphMaker::finish()
{
  // Add all waypoints, so that lonely points also appear in the triangulation
  if (ed->getWaypointsEditor()) {
    for (std::vector<PathEditor::WayPoint>::const_iterator it = ed->getWaypointsEditor()->waypoints.begin();
         it != ed->getWaypointsEditor()->waypoints.end();
         ++it)
    {
      sg->addPoint(it->pos);
    }
  }
}

void SectionsEditor::SimpleSectionsNode::handleSection(const top10::track::Section& track_section)
{    
  std::vector<top10::math::Vector> left, right;
          
  track_section.vectorize(track_section.getLeftDist1(), track_section.getLeftDist2(), max_angle, min_dist, left);
  track_section.vectorize(track_section.getRightDist1(), track_section.getRightDist2(), max_angle, min_dist, right);
  
  lefts.push_back(left);
  rights.push_back(right);
}

void SectionsEditor::SimpleSectionsNode::finish()
{
  // Nothing!
}

void SectionsEditor::SimpleSectionsNode::renderGL(const top10::graphX::RenderingFeatures& features,
                                                  const top10::graphX::RenderState& state,
                                                  const top10::graphX::CameraNode& camera) const
{
  for (std::list< std::vector< top10::math::Vector > >::const_iterator outline_it = lefts.begin();
      outline_it != lefts.end();
      ++outline_it)
  {
    // Draw the outlines
    glBegin(GL_LINE_STRIP);
    for (std::vector< top10::math::Vector >::const_iterator vec_it = outline_it->begin(); vec_it != outline_it->end(); ++vec_it) {
      glVertex3f(vec_it->x, vec_it->y, vec_it->z);
    }
    glEnd();
    
    // Draw the ends
    for (std::vector< top10::math::Vector >::const_iterator vec_it = outline_it->begin(); vec_it != outline_it->end(); ++vec_it) {
      drawPoint(*vec_it);
    }

  }
  
  for (std::list< std::vector< top10::math::Vector > >::const_iterator outline_it = rights.begin();
      outline_it != rights.end();
      ++outline_it)
  {
    glBegin(GL_LINE_STRIP);
    for (std::vector< top10::math::Vector >::const_iterator vec_it = outline_it->begin(); vec_it != outline_it->end(); ++vec_it) {
      glVertex3f(vec_it->x, vec_it->y, vec_it->z);
    }
    glEnd();
    
    // Draw the ends
    for (std::vector< top10::math::Vector >::const_iterator vec_it = outline_it->begin(); vec_it != outline_it->end(); ++vec_it) {
      drawPoint(*vec_it);
    }

  }
}

void SectionsEditor::SimpleSectionsNode::drawPoint(top10::math::Vector v) const
{
  glBegin(GL_LINES);
  glVertex3f(v.x-0.2, v.y, v.z);
  glVertex3f(v.x+0.2, v.y, v.z);
  glVertex3f(v.x, v.y, v.z-0.2);
  glVertex3f(v.x, v.y, v.z+0.2);
  glEnd();
}

void SectionsEditor::CursorNode::renderGL(const top10::graphX::RenderingFeatures& features,
                                          const top10::graphX::RenderState& state,
                                          const top10::graphX::CameraNode& camera) const
{
  // No current control point
  if (ed->getCurrentControlIndex() == -1) return;

  // Too few points in the current path
  if (ed->current_path->pts.size() <= 1) return;

  if (ed->getCurrentControlIndex() > 0 || ed->getLoops()) {
    drawSectionPart(ed->prevControl(ed->getCurrentControlIndex()), ed->getCurrentControlIndex(), 0.5, 1.0);
  }
  if (ed->getCurrentControlIndex() < ed->getLastControlIndex() || ed->getLoops()) {
    drawSectionPart(ed->getCurrentControlIndex(), ed->nextControl(ed->getCurrentControlIndex()), 0.0, 0.5);
  }
}

void SectionsEditor::CursorNode::drawSectionPart(int idx1, int idx2, float t1, float t2) const
{
  // Control points before and after the current control point, and the current one itself
  SectionsEditor::ControlPoint ctrl1, ctrl2;
  ctrl1 = ed->getControl(idx1);
  ctrl2 = ed->getControl(idx2);

  // Build the curve ending in the current control point
  try {
    top10::math::Hermite curve;
    top10::track::Waypoint wp1, wp2;
    wp1 = ed->getWaypoint(ctrl1);
    wp2 = ed->getWaypoint(ctrl2);
    
    curve.setEnd1(wp1.getPos());
    curve.setEnd2(wp2.getPos());
    curve.setTangent1(wp1.getTangent() * ctrl1.tg_size_1);
    curve.setTangent2(wp2.getTangent() * ctrl2.tg_size_2);

    // Create the section
    top10::track::Section track_section;
    track_section.setCurve(curve);
    
    track_section.setLeftDist1(-ctrl1.left_road_width);
    track_section.setRightDist1(ctrl1.right_road_width);
    track_section.setBankAngle1(M_PI*wp1.getBankAngle()/180.0);
    
    track_section.setLeftDist2(-ctrl2.left_road_width);
    track_section.setRightDist2(ctrl2.right_road_width);
    track_section.setBankAngle2(M_PI*wp2.getBankAngle()/180.0);
  
    // vectorize
    std::vector<top10::math::Vector> left, right;
    track_section.vectorize(1.1*track_section.getLeftDist1(), 1.1*track_section.getLeftDist2(),
                            max_angle, min_dist, left, t1, t2);
    track_section.vectorize(1.1*track_section.getRightDist1(), 1.1*track_section.getRightDist2(),
                            max_angle, min_dist, right, t1, t2);
  
    // render
    glBegin(GL_LINE_STRIP);
    for (std::vector<top10::math::Vector>::const_iterator it = left.begin(); it != left.end(); ++it)
      glVertex3f(it->x, it->y, it->z);
    glEnd();
  
    glBegin(GL_LINE_STRIP);
    for (std::vector<top10::math::Vector>::const_iterator it = right.begin(); it != right.end(); ++it)
      glVertex3f(it->x, it->y, it->z);
    glEnd();
  }
  catch(NoSuchWaypoint) {}
}

}
}
