/*
  Top10, a racing simulator
  Copyright (C) 2000-2005  Johann Deneux
  
  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.
  
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
  
  Authors can be contacted at following electronic addresses:
  Johann Deneux: johann.deneux@it.uu.se
*/

#ifndef TOP10_TRACKEDPATHEDITOR_HH
#define TOP10_TRACKEDPATHEDITOR_HH

#include "Drawable.hh"
#include "MainCamera.hh"
#include "graphX/GroupNode.hh"
#include <tinyxml.h>

class fntFont;

namespace top10 {

namespace tracked {

class SectionsEditor;
  
//! Manage a set of waypoints
/*! This class was initially used to make a circular path, used to manually measure the length of a track.
  Handling paths, i.e. connections between way points is handled in the SectionsEditor (which uses PathEditor) */
class PathEditor : public top10::tracked::Drawable
{
public:
  struct WayPoint {
    //! Unique name of this waypoint, used by users to connect way points
    std::string name;
    //! Unique id of this point. Unlike the name, it cannot change, making it suitable for external refs
    int id;
    //! Position of the waypoint
    top10::math::Vector pos;
    //! Tangent at the waypoint
    top10::math::Vector tg;
    //! Secondary tangent, useful for crossings
    top10::math::Vector tg2;
    //! Bank (twist) angle, in degrees (0 = horizontal)
    float bank_angle;
  };
  
  //! Base class for all classes which wish to be notified when something changes
  class Listener: public top10::util::RefCount {
  public:
    virtual void notifyAdd(const top10::tracked::PathEditor::WayPoint&);
    virtual void notifyRemove(top10::tracked::PathEditor::WayPoint);
    virtual void notifyMove(const top10::tracked::PathEditor::WayPoint&);
    virtual void notifyTangent(const top10::tracked::PathEditor::WayPoint&);
    virtual void notifyTangent2(const top10::tracked::PathEditor::WayPoint&);
    virtual void notifyBanking(const top10::tracked::PathEditor::WayPoint&);
    virtual void notifyName(const top10::tracked::PathEditor::WayPoint&);
    
    virtual ~Listener() {}
  };
  typedef top10::util::Ref< Listener > ListenerRef;
  typedef std::vector< ListenerRef > ListenerRefs;

public:
  PathEditor();

  int loadXml(const TiXmlElement* xml_node);
  int saveXml(TiXmlElement* xml_node) const;
  void clearState();
  
  inline void setFont(fntFont* f) { font = f; }
  inline void setCamera(MainCamera* p) { camera = p; }
  inline void setSectionsEditor(SectionsEditor* p) { sections_ed = p; }
  
  void addListener(Listener*);
  void removeListener(Listener*);
  
  //! Use the current aimed point
  void pick();
    
  //! Delete the current waypoint
  void remove();

  //! Insert a waypoint after the current one
  void addNew();

  //! Move to the next waypoint
  void next();
  void previous();
  void gotoIdx(int idx);

  //! Return the index of a point by its name
  /*! \return -1 if not found */
  int findNamed(std::string) const;
  
  //! Search by id
  /*! \return -1 if not found */
  int findById(int id) const;
  
  inline int getCurrentIndex() const { return current_cp; }
  int getNextPoint(int) const;
  int getPrevPoint(int) const;
  
  WayPoint getCurrentWayPoint() const;
  void setCurrentWayPoint(WayPoint);
  
  //! Return the position current waypoint
  top10::math::Vector getCurrentPos() const;
  //! Return the tangent at this point
  top10::math::Vector getCurrentTangent() const;
  //! Return the secondary tangent at this point
  top10::math::Vector getCurrentTangent2() const;
  //! Return the bank angle at the current waypoint
  float getCurrentBankAngle() const;
  //! Return the name of this waypoint
  std::string getCurrentName() const;
  
  void setCurrentPos(top10::math::Vector);
  void setCurrentTangent(top10::math::Vector, int id = -1);
  void setCurrentTangent2(top10::math::Vector, int id = -1);
  void setCurrentBankAngle(float);
  void setCurrentName(std::string);
  
  void getMinMax(double* x_min, double* z_min, double* x_max, double* z_max) const;
  
  inline std::string getOrigin() const { return "PathEditor"; }
  
public:
  std::vector<WayPoint> waypoints;
  
  // Controls the appearance of labels.
  unsigned char label_r;
  unsigned char label_g;
  unsigned char label_b;
  double label_point_size;

private:
  ListenerRefs listeners;
  fntFont* font; 

  //! Draw all waypoints
  class PathNode: public top10::graphX::LeafNode {
  public:
    PathNode(PathEditor* ed): ed(ed), mark_size(0.5), unit_size(3.0) {}
    void renderGL(const top10::graphX::RenderingFeatures&, const top10::graphX::RenderState&, const top10::graphX::CameraNode&) const;    
  private:
    PathEditor* ed;
    float mark_size;
    float unit_size;
  };
  
  //! Draw the cursor on the current waypoint
  class CursorNode: public top10::graphX::LeafNode {
  public:
    CursorNode(PathEditor* ed): ed(ed) {}
    void renderGL(const top10::graphX::RenderingFeatures&, const top10::graphX::RenderState&, const top10::graphX::CameraNode&) const;    
  private:
    PathEditor* ed;
  };
  
  top10::util::Ref<PathNode> path_node;
  top10::util::Ref<CursorNode> cursor_node;
  top10::util::Ref<top10::graphX::GroupNode> labels_group;

  int current_cp;
  MainCamera* camera;
  SectionsEditor* sections_ed;
  int last_id;
  
  bool isCorrect(int current_cp) const;
  TiXmlElement* find(TiXmlNode* xml_node, int search_id) const;
  void setVectorXml(TiXmlElement* wp_el, const char* value, top10::math::Vector v) const;
  
  //! If the name of the current waypoint is of the form name_number, return name and number.
  /*! Otherwise, return "WP" and 1 */
  void getCurrentBasename(std::string* name, int* number) const;

  //! Rebuild the scene graph showing labels
  void makeLabels();
};

}

}

#endif
