/*
  Top10, a racing simulator
  Copyright (C) 2000-2005  Johann Deneux
  
  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.
  
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
  
  Authors can be contacted at following electronic addresses:
  Johann Deneux: johann.deneux@it.uu.se
*/

#include "PathEditor.hh"
#include "SectionsEditor.hh"
#include "util/Log.hh"
#include "util/strconv.hh"
#include "graphX/TextNode.hh"
#include "graphX/BillBoardNode.hh"
#include "graphX/TransformNode.hh"

#include <plib/fnt.h>
#include <sstream>

using top10::util::Log;

namespace top10 {
namespace tracked {

PathEditor::PathEditor():
  Drawable("waypoints"),
  label_r(255),
  label_g(100),
  label_b(200),
  label_point_size(2.0),
  font(0), camera(0), sections_ed(0), last_id(0)
{
  path_node = new PathNode(this);
  cursor_node = new CursorNode(this);
  addChild(path_node.getPtr());
  addChild(cursor_node.getPtr());
  current_cp = -1;
}

void PathEditor::addListener(Listener* listener)
{
  listeners.push_back(ListenerRef(listener));
}

void PathEditor::removeListener(Listener* listener)
{
  ListenerRefs tmp;
  tmp.reserve(listeners.size());
  for (ListenerRefs::const_iterator it = listeners.begin(); it != listeners.end(); ++it)
    if (it->getPtr() != listener)
      tmp.push_back(*it);

  listeners.swap(tmp);
}

void PathEditor::getMinMax(double* x_min, double* z_min, double* x_max, double* z_max) const
{
  for (std::vector<WayPoint>::const_iterator it = waypoints.begin(); it != waypoints.end(); ++it)
  {
    if (it->pos.x > *x_max) *x_max = it->pos.x;
    if (it->pos.x < *x_min) *x_min = it->pos.x;
    if (it->pos.z > *z_max) *z_max = it->pos.z;
    if (it->pos.z < *z_min) *z_min = it->pos.z;
  }
}

void PathEditor::pick()
{
  assert(camera);
  assert(isCorrect(current_cp));

  if (current_cp != -1)
  {
    using top10::math::Vector;
    using top10::math::Plane;

    Vector pt = waypoints.at(current_cp).pos;
    const Vector dir = camera->getView().getDirection();
    const Vector pos = camera->getView().getCenter();
    double dist = 0.0;
    const Plane plane( Vector(0.0, 1.0, 0.0), pt.y );
    bool b = intersectRay(plane, pos, dir, pt, dist);

    if (b)
    {
      waypoints.at(current_cp).pos = pt;
    
      // Notify listeners
      for (int i=0; i<listeners.size(); ++i) listeners[i]->notifyMove(waypoints.at(current_cp));
    
      makeLabels();
    }
  }
  else Log::getSingle()->send(Log::Error, getOrigin(), "You must select a waypoint first");
}

void PathEditor::setCurrentPos(top10::math::Vector v)
{
  assert(isCorrect(current_cp));
  
  if (current_cp != -1) {
    waypoints.at(current_cp).pos = v;
    
    // Notify listeners
    for (int i=0; i<listeners.size(); ++i) listeners[i]->notifyMove(waypoints.at(current_cp));

    makeLabels();
  }
  else Log::getSingle()->send(Log::Error, getOrigin(), "You must select a waypoint first");
}

void PathEditor::setCurrentTangent(top10::math::Vector v, int id)
{
  int cp;
  if (id == -1) cp = current_cp;
  else
  {
    cp = findById(id);
    if (cp == -1) {
      Log::getSingle()->send(Log::Critical, getOrigin(), "Could not find waypoint with it "+top10::util::toString(id));
      return;
    }
  }

  assert(isCorrect(cp));
  
  if (cp != -1) {
    waypoints.at(cp).tg = v;
    
    // Notify listeners
    for (int i=0; i<listeners.size(); ++i) listeners[i]->notifyTangent(waypoints.at(cp));

  }
  else Log::getSingle()->send(Log::Error, getOrigin(), "You must select a waypoint first");
}

void PathEditor::setCurrentTangent2(top10::math::Vector v, int id)
{
  int cp;
  if (id == -1) cp = current_cp;
  else
  {
    cp = findById(id);
    if (cp == -1) {
      Log::getSingle()->send(Log::Critical, getOrigin(), "Could not find waypoint with it "+top10::util::toString(id));
      return;
    }
  }

  assert(isCorrect(cp));
  
  if (cp != -1) {
    waypoints.at(cp).tg2 = v;
    
    // Notify listeners
    for (int i=0; i<listeners.size(); ++i) listeners[i]->notifyTangent2(waypoints.at(cp));

  }
  else Log::getSingle()->send(Log::Error, getOrigin(), "You must select a waypoint first");
}

void PathEditor::setCurrentBankAngle(float f)
{
  assert(isCorrect(current_cp));
  
  if (current_cp != -1) {
    waypoints.at(current_cp).bank_angle = f;
    
    // Notify listeners
    for (int i=0; i<listeners.size(); ++i) listeners[i]->notifyBanking(waypoints.at(current_cp));
    
  }
  else Log::getSingle()->send(Log::Error, getOrigin(), "You must select a waypoint first");
}

void PathEditor::setCurrentName(std::string s)
{
  assert(isCorrect(current_cp));
  
  if (current_cp != -1) {
    waypoints.at(current_cp).name = s;
    
    // Notify listeners
    for (int i=0; i<listeners.size(); ++i) listeners[i]->notifyName(waypoints.at(current_cp));
  
    makeLabels();
  }
  else Log::getSingle()->send(Log::Error, getOrigin(), "You must select a waypoint first");
}

void PathEditor::addNew()
{
  assert(isCorrect(current_cp));
  assert(sections_ed);
  
  WayPoint new_one;
  new_one.id = ++last_id;
  new_one.pos = top10::math::Vector(0.0, 0.0, 0.0);
  new_one.tg = top10::math::Vector(1.0, 0.0, 0.0);
  new_one.tg2 = top10::math::Vector(0.0, 0.0, 1.0);
  new_one.bank_angle = 0.0;
  // Generate a unique name
  int i=1;
  std::string basename;
  getCurrentBasename(&basename, &i);
  do {
    std::ostringstream buf;
    buf<<basename<<"_"<<i++;
    new_one.name = buf.str();
  } while (findNamed(new_one.name) != -1);
  
  waypoints.insert(waypoints.begin()+(current_cp+1), new_one);
  next();
  pick();
  
  // Add it to the current path in the sections editor
  //sections_ed->addSection();
  
  Log::getSingle()->send(Log::Info, getOrigin(), "Created new waypoint " + new_one.name);
  
  // Notify listeners
  for (int i=0; i<listeners.size(); ++i) listeners[i]->notifyAdd(new_one);

  // Set the tangent, using the previous waypoint, if any
  if (waypoints.size() >= 2)
  {
    previous();
    top10::math::Vector pos_prev = getCurrentPos();
    next();
    top10::math::Vector pos_current = getCurrentPos();
    top10::math::Vector diff = pos_current - pos_prev;
    double sz = diff.size();
    if (sz >= SMALL_VECTOR)
    {
      diff /= sz;
      setCurrentTangent(diff);
      setCurrentTangent2(top10::math::Vector(diff.z, diff.y, -diff.x));
    }
  }

  makeLabels();
}

void PathEditor::getCurrentBasename(std::string* name_out, int* i_out) const
{
  *name_out = "WP";
  *i_out = 1;
  
  if (current_cp == -1) return;
  
  // Name of the current waypoint
  std::string name = waypoints.at(current_cp).name;
  // Look for _
  int pos = name.find("_");
  if (pos < 0 || pos >= name.size()) {
    // Not found, return "name" 1
    *name_out = name;
    return;
  }
  // Extract the part before the _
  *name_out = name.substr(0, pos);
  // Parse the number after _
  if (pos < name.size()-1) {
    std::istringstream buf(name.substr(pos+1, name.size()-pos-1));
    buf>>*i_out;
  }
  
}
void PathEditor::remove()
{
  assert(isCorrect(current_cp));
  
  if (current_cp != -1) {
    // Notify listeners
    for (int i=0; i<listeners.size(); ++i) listeners[i]->notifyRemove(waypoints.at(current_cp));
    
    waypoints.erase(waypoints.begin()+current_cp);
    
    makeLabels();
    
    --current_cp;
  }
  else Log::getSingle()->send(Log::Error, getOrigin(), "You must select a waypoint first");
  
  assert(isCorrect(current_cp));
}

void PathEditor::next()
{
  assert(isCorrect(current_cp));
  
  if (current_cp+1 == (int)waypoints.size())
  {
    if (!waypoints.empty()) current_cp = 0;
  }
  else ++current_cp;
  
  assert(isCorrect(current_cp));
}

void PathEditor::previous()
{
  assert(isCorrect(current_cp));
  
  if (current_cp == 0)
  {
    if (!waypoints.empty()) current_cp = (int)waypoints.size() -1;
  }
  else --current_cp;
  
  assert(isCorrect(current_cp));
}

void PathEditor::gotoIdx(int idx)
{
  assert(isCorrect(current_cp));

  if (idx < 0 || idx >= (int)waypoints.size()) return;
  current_cp = idx;

  assert(isCorrect(current_cp));
}

int PathEditor::getNextPoint(int i) const
{
  if (waypoints.empty()) return -1;
  if (i == (int)waypoints.size()-1) return 0;
  return i+1;
}

int PathEditor::getPrevPoint(int i) const
{
  if (waypoints.empty()) return -1;
  if (i == 0) return (int)waypoints.size()-1;
  return i-1;
}

int PathEditor::findNamed(std::string name) const
{
  int id = 0;
  for (std::vector<WayPoint>::const_iterator it = waypoints.begin(); it != waypoints.end(); ++it, ++id) {
    if (it->name == name) return id;
  }
  return -1;
}
  
int PathEditor::findById(int id) const
{
  int i = 0;
  for (std::vector<WayPoint>::const_iterator it = waypoints.begin(); it != waypoints.end(); ++it, ++i) {
    if (it->id == id) return i;
  }
  return -1;
}

PathEditor::WayPoint PathEditor::getCurrentWayPoint() const
{
  WayPoint ret;
  ret.id = 0;
  ret.bank_angle = 0.0;
  
  assert(isCorrect(current_cp));
  if (current_cp != -1)
    ret = waypoints.at(current_cp);
  return ret;
}

void PathEditor::setCurrentWayPoint(WayPoint wp)
{
  assert(isCorrect(current_cp));
  if (current_cp != -1) {
    waypoints.at(current_cp) = wp;
  }
  else Log::getSingle()->send(Log::Error, getOrigin(), "You must select a waypoint first");
}
  
  
top10::math::Vector PathEditor::getCurrentPos() const
{
  assert(isCorrect(current_cp));
  if (current_cp == -1) return top10::math::Vector();
  else return waypoints.at(current_cp).pos;
}

top10::math::Vector PathEditor::getCurrentTangent() const
{
  assert(isCorrect(current_cp));
  if (current_cp == -1) return top10::math::Vector();
  else return waypoints.at(current_cp).tg;
}

top10::math::Vector PathEditor::getCurrentTangent2() const
{
  assert(isCorrect(current_cp));
  if (current_cp == -1) return top10::math::Vector();
  else return waypoints.at(current_cp).tg2;
}

float PathEditor::getCurrentBankAngle() const
{
  assert(isCorrect(current_cp));
  if (current_cp == -1) return 0.0;
  else return waypoints.at(current_cp).bank_angle;
}

std::string PathEditor::getCurrentName() const
{
  assert(isCorrect(current_cp));
  if (current_cp == -1) return std::string("");
  else return waypoints.at(current_cp).name;
}

void PathEditor::PathNode::renderGL(const top10::graphX::RenderingFeatures&,
  const top10::graphX::RenderState&,
  const top10::graphX::CameraNode&) const
{
  using top10::math::Vector;

  assert(ed->isCorrect(ed->current_cp));
  
  // For each waypoint, draw a mark at its location and a normalized tangent
  for (int cp = 0; cp != (int)ed->waypoints.size(); ++cp) {
    Vector v = ed->waypoints.at(cp).pos;
    glBegin(GL_LINES);
      // The mark
      glVertex3f(v.x - mark_size, v.y, v.z);
      glVertex3f(v.x + mark_size, v.y, v.z);
      glVertex3f(v.x, v.y, v.z - mark_size);
      glVertex3f(v.x, v.y, v.z + mark_size);
      
      // The tangent
      Vector v2 = ed->waypoints.at(cp).tg;
      double s = v2.size();
      if (s >= SMALL_VECTOR) {
        v2 /= s;
        glVertex3f(v.x, v.y, v.z);
        v2 = v + unit_size*v2;
        glVertex3f(v2.x, v2.y, v2.z);
      }
    glEnd();
  }
}

void PathEditor::CursorNode::renderGL(const top10::graphX::RenderingFeatures&,
  const top10::graphX::RenderState&,
  const top10::graphX::CameraNode&) const
{
  using top10::math::Vector;

  assert(ed->isCorrect(ed->current_cp));

  if (ed->current_cp == -1) return;
  
  // Mark the current waypoint
  top10::math::Vector p = ed->getCurrentPos();
  
  glBegin(GL_LINES);
  glVertex3f(p.x-1, p.y+2, p.z);
  glVertex3f(p.x+1, p.y+2, p.z);
  glVertex3f(p.x, p.y+2, p.z-1);
  glVertex3f(p.x, p.y+2, p.z+1);
  glEnd();
}

//! Check the checkpoint index is in range
bool PathEditor::isCorrect(int cp) const
{
  return (cp >= -1) && (cp+1 <= (int)waypoints.size());
}

int PathEditor::loadXml(const TiXmlElement* xml_node)
{
  assert(xml_node);
  
  if (xml_node->Value() != std::string("waypoints")) {
    Log::getSingle()->send(Log::Error, getOrigin(), std::string("Expected <wayppoints>, got ")+xml_node->Value()+
      " on line "+top10::util::toString(xml_node->Row())+
      " column "+top10::util::toString(xml_node->Column()));
    return -1;
  }
  
  std::vector<WayPoint> tmp_waypoints;
  // Iterate through the children
  const TiXmlElement* wp_node = xml_node->FirstChildElement("waypoint");
  while (wp_node) {
    try {
      WayPoint new_one;
      
      const char* attr = wp_node->Attribute("name");
      if (attr) new_one.name = attr;
      else throw std::string("Missing name attribute");
      
      int status = wp_node->QueryIntAttribute("id", &new_one.id);
      if (status == TIXML_WRONG_TYPE) throw std::string("Wrong type, expected integer");
      else if (status == TIXML_NO_ATTRIBUTE) throw std::string("Missing id attribute");
      if (new_one.id > last_id) last_id = new_one.id;
      
      status = wp_node->QueryFloatAttribute("bank", &new_one.bank_angle);
      if (status == TIXML_WRONG_TYPE) throw std::string("Wrong type, expected float");
      else if (status == TIXML_NO_ATTRIBUTE) new_one.bank_angle = 0.0;
      
      const TiXmlElement* wp_pos = wp_node->FirstChildElement("position");
      if (wp_pos == 0) throw std::string("Missing position element");
      new_one.pos.loadXml(wp_pos);
      
      const TiXmlElement* wp_tg = wp_node->FirstChildElement("tangent");
      if (wp_tg == 0) logXmlNodeError("Missing primary tangent element", wp_node);
      else new_one.tg.loadXml(wp_tg);

      const TiXmlElement* wp_tg2 = wp_tg->NextSiblingElement("tangent2");
      if (wp_tg2 == 0) logXmlNodeError("Missing secondary tangent element", wp_node);
      else new_one.tg2.loadXml(wp_tg2);

      tmp_waypoints.push_back(new_one);
    }
    catch (const std::string& err) {
      Log::getSingle()->send(Log::Error, getOrigin()+"/XmlParser", err +
          " line "+top10::util::toString(wp_node->Row())+
          " column "+top10::util::toString(wp_node->Column()));
    }
    wp_node = wp_node->NextSiblingElement("waypoint");
  }
  
  waypoints.swap(tmp_waypoints);
  current_cp = -1;
  
  /* Notify the removal of all old waypoints. */
  for (std::vector<WayPoint>::const_iterator it = tmp_waypoints.begin();
    it != tmp_waypoints.end();
    ++it)
  {
    for (int i=0; i<listeners.size(); ++i)
      listeners[i]->notifyRemove(*it);
  }

  /* Notify the addition of all new waypoints. */
  for (std::vector<WayPoint>::const_iterator it = waypoints.begin();
    it != waypoints.end();
    ++it)
  {
    for (int i=0; i<listeners.size(); ++i)
      listeners[i]->notifyAdd(*it);
  }

  makeLabels();

  return 0;
}

/*!
  \param xml_node element. Its value must be "waypoints".
  */
int PathEditor::saveXml(TiXmlElement* xml_node) const
{
  assert(xml_node);
  
  if (xml_node->Value() != std::string("waypoints")) {
    Log::getSingle()->send(Log::Error, getOrigin(), std::string("Expected <wayppoints>, got ")+xml_node->Value()+
      " on line "+top10::util::toString(xml_node->Row())+
      " column "+top10::util::toString(xml_node->Column()));
    return -1;
  }
  
  // Remove all elements whose ids are no longer in waypoints
  TiXmlElement* wp_el = xml_node->FirstChildElement("waypoint");
  while (wp_el) {
    int id;
    int status = wp_el->QueryIntAttribute("id", &id);
    if (status != TIXML_SUCCESS) {
      TiXmlElement* wp_el2 = wp_el->NextSiblingElement("waypoint");
      xml_node->RemoveChild(wp_el);
      wp_el = wp_el2;
    }
    else {
      bool found = false;
      for (std::vector<WayPoint>::const_iterator f_it = waypoints.begin(); !found && f_it != waypoints.end(); ++f_it) {
        if (f_it->id == id) found = true;
      }
      if (!found) {
        TiXmlElement* wp_el2 = wp_el->NextSiblingElement("waypoint");
        xml_node->RemoveChild(wp_el);
        wp_el = wp_el2;
      }
      else wp_el = wp_el->NextSiblingElement("waypoint");
    }
  }
  
  // Update or add elements for all points in waypoints
  for (std::vector<WayPoint>::const_iterator wp_it = waypoints.begin(); wp_it != waypoints.end(); ++wp_it) {
    bool must_insert = false;
    TiXmlElement new_el("waypoint");
    // Try to find an xml element with value "waypoint" whose id is wp_it->id
    TiXmlElement* wp_el = find(xml_node, wp_it->id);
    if (wp_el == 0) {
      must_insert = true;
      wp_el = &new_el;
    }
    wp_el->SetAttribute("name", wp_it->name);
    wp_el->SetAttribute("id", wp_it->id);
    wp_el->SetDoubleAttribute("bank", wp_it->bank_angle);
    setVectorXml(wp_el, "position", wp_it->pos);
    setVectorXml(wp_el, "tangent", wp_it->tg);
    setVectorXml(wp_el, "tangent2", wp_it->tg2);
    if (must_insert) xml_node->InsertEndChild(*wp_el);
  }
  
  return 0;
}

TiXmlElement* PathEditor::find(TiXmlNode* xml_node, int search_id) const
{
  TiXmlElement* wp_el = xml_node->FirstChildElement("waypoint");
  while (wp_el) {
    int id;
    int status = wp_el->QueryIntAttribute("id", &id);
    if (status == TIXML_SUCCESS && id == search_id) return wp_el;
    wp_el = wp_el->NextSiblingElement("waypoint");
  }
  return 0;
}

void PathEditor::setVectorXml(TiXmlElement* wp_el, const char* value, top10::math::Vector v) const
{
  bool must_insert = false;
  TiXmlElement new_el(value);
  // Try to find an xml element with the right value
  TiXmlElement* pos_el = wp_el->FirstChildElement(value);
  if (pos_el == 0) {
    must_insert = true;
    pos_el = &new_el;
  }
  
  v.saveXml(pos_el);
  if (must_insert) wp_el->InsertEndChild(*pos_el);
}

void PathEditor::clearState( )
{
  waypoints.clear();
  current_cp = -1;
  makeLabels();
}

void PathEditor::Listener::notifyAdd(const top10::tracked::PathEditor::WayPoint&) {}
void PathEditor::Listener::notifyRemove(top10::tracked::PathEditor::WayPoint) {}
void PathEditor::Listener::notifyBanking(const top10::tracked::PathEditor::WayPoint&) {}
void PathEditor::Listener::notifyMove(const top10::tracked::PathEditor::WayPoint&) {}
void PathEditor::Listener::notifyName(const top10::tracked::PathEditor::WayPoint&) {}
void PathEditor::Listener::notifyTangent(const top10::tracked::PathEditor::WayPoint&) {}
void PathEditor::Listener::notifyTangent2(const top10::tracked::PathEditor::WayPoint&) {}

void PathEditor::makeLabels()
{
  removeChild(labels_group.getPtr());
  labels_group.discard();
  labels_group = new top10::graphX::GroupNode;

  for (std::vector<WayPoint>::const_iterator it = waypoints.begin();
    it != waypoints.end();
    ++it)
  {      
    top10::graphX::TransformNode* translation = new top10::graphX::TransformNode;
    top10::math::Translation4 T(it->pos);
    translation->setToWorld(T);

    top10::graphX::BillBoardNode* billboard = new top10::graphX::BillBoardNode;

    top10::graphX::TextNode* label = new top10::graphX::TextNode;
    label->setFont(font);
    label->setText(it->name);
    label->setPointSize(label_point_size);

    top10::graphX::MaterialNode* label_color = new top10::graphX::MaterialNode;
    label_color->r = label_r;
    label_color->g = label_g;
    label_color->b = label_b;

    billboard->addChild(label_color);
    translation->addChild(billboard);
    label_color->addChild(label);
    labels_group->addChild(translation);
  }

  addChild(labels_group.getPtr());
}

}
}

