/*
  Top 10, a racing simulator
  Copyright (C) 2003,2005,2006  Johann Deneux
  
  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.
  
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
  
  Authors can be contacted at following electronic addresses:
  Johann Deneux: johann.deneux@gmail.com
*/
#include "MeshInstanceEditor.hh"
#include "util/NameFinder.hh"
#include "util/Log.hh"
#include "util/findpred.hh"

using top10::util::Log;

top10::tracked::MeshInstanceEditor::MeshInstanceEditor( ):
  Drawable("mesh_instances"), mesh_lib_ed(0)
{
  clearState();
}

void top10::tracked::MeshInstanceEditor::setMeshEditor( top10::helpers::MeshEditor* ed)
{
  mesh_lib_ed = ed;
}

top10::graphX::Node* top10::tracked::MeshInstanceEditor::getAllNodes() const
{
  top10::graphX::GroupNode* ret = new top10::graphX::GroupNode;
  for (InstanceList::const_iterator it = instances.begin(); it != instances.end(); ++it)
  {
    ret->addChild(it->transform_node.getPtr());
  }
  return ret;
}

std::vector<top10::tracked::MeshInstanceEditor::Item>
top10::tracked::MeshInstanceEditor::getItems() const
{
  std::vector<Item> ret;
  ret.reserve(instances.size());

  for (InstanceList::const_iterator it = instances.begin(); it != instances.end(); ++it)
  {
    Item item;
    item.m_node = it->transform_node.getPtr();
    item.m_collides = !(mesh_lib_ed->getFlags(it->mesh_id) & top10::helpers::MeshEditor::NO_COLLISION);

    ret.push_back(item);
  }
  return ret;
}

void top10::tracked::MeshInstanceEditor::gotoInstance( int idx )
{
  InstanceList::iterator it = instances.begin();
  for (int i=0; i<idx && it != instances.end(); ++it);
  current_inst = it;
  updateMarker();
}

void top10::tracked::MeshInstanceEditor::updateMarker()
{
  removeChild(marker_node.getPtr());
  marker_node = 0;
  if (current_inst != instances.end())
  {
    marker_node = new top10::graphX::MeshMarker;
    marker_node->addChild(current_inst->transform_node.getPtr());
  }
  addChild(marker_node.getPtr());
}

void top10::tracked::MeshInstanceEditor::gotoNext()
{
  if (current_inst != instances.end())
  {
    ++current_inst;
    if (current_inst == instances.end())
      current_inst = instances.begin();
  }
  else
    current_inst = instances.begin();

  updateMarker();
}

void top10::tracked::MeshInstanceEditor::gotoPrev()
{
  if (current_inst != instances.begin())
    --current_inst;
  else
    current_inst = instances.end();

  updateMarker();
}

int top10::tracked::MeshInstanceEditor::getPos() const
{
  int pos = 0;
  InstanceList::const_iterator it = instances.begin();
  for (; it != current_inst && it != instances.end(); ++it, ++pos);

  if (it == instances.end())
    return -1;
  else
    return pos;
}

void top10::tracked::MeshInstanceEditor::addInstance( )
{
  // The new instance to add
  MeshInstance new_one;
  new_one.mesh_id = -1;
  new_one.rotation = 0.0;
  new_one.transform_node = new top10::graphX::TransformNode;
  addChild(new_one.transform_node.getPtr());

  // Add the new instance
  instances.push_back(new_one);

  current_inst = instances.end();
  current_inst--;

  updateMarker();
}

void top10::tracked::MeshInstanceEditor::deleteInstance( )
{
  if (current_inst == instances.end()) return;
  removeChild(current_inst->transform_node.getPtr());
  instances.erase(current_inst);
  current_inst = instances.begin();

  updateMarker();
}

void top10::tracked::MeshInstanceEditor::setMeshName( std::string name )
{
  if (current_inst == instances.end()) return;
  assert(mesh_lib_ed);
  current_inst->mesh_id = mesh_lib_ed->findId(name);
  setMeshNode();
}

void top10::tracked::MeshInstanceEditor::setInstanceTranslation( const top10::math::Vector& T )
{
  if (current_inst == instances.end()) return;
  current_inst->translation = T;
  current_inst->transform_node->setToWorld(getInstanceTransform());
}

top10::math::Vector top10::tracked::MeshInstanceEditor::getInstanceTranslation( ) const
{
  if (current_inst == instances.end()) return top10::math::Vector();
  return current_inst->translation;
}

void top10::tracked::MeshInstanceEditor::setInstanceRotation( double degrees )
{
  if (current_inst == instances.end()) return;
  current_inst->rotation = degrees;
  current_inst->transform_node->setToWorld(getInstanceTransform());
}

double top10::tracked::MeshInstanceEditor::getInstanceRotation( ) const
{
  if (current_inst == instances.end()) return 0.0;
  return current_inst->rotation;
}

std::string top10::tracked::MeshInstanceEditor::getMeshName( ) const
{
  if (current_inst == instances.end()) return "";
  assert(mesh_lib_ed);
  return mesh_lib_ed->getMeshName(current_inst->mesh_id);
}

std::list<std::string> top10::tracked::MeshInstanceEditor::getAllMeshNames() const
{
  return mesh_lib_ed->getMeshNames();
}

int top10::tracked::MeshInstanceEditor::getMeshId( ) const
{
  if (current_inst == instances.end()) return -1;
  return current_inst->mesh_id;
}

top10::math::Matrix4 top10::tracked::MeshInstanceEditor::getInstanceTransform( ) const
{
  if (current_inst == instances.end()) return top10::math::Matrix4();
  top10::math::Rotation3 R(M_PI*current_inst->rotation/180.0, top10::math::Vector(0.0, 1.0, 0.0));
  top10::math::Matrix4 ret = R;
  ret = top10::math::Translation4(current_inst->translation) * ret;
  return ret;
}

std::string top10::tracked::MeshInstanceEditor::getOrigin( ) const
{
  return "MeshInstanceEditor";
}

void top10::tracked::MeshInstanceEditor::clearState( )
{
  for (InstanceList::iterator it = instances.begin(); it != instances.end(); ++it)
  {
    removeChild(it->transform_node.getPtr());
  }
  instances.clear();
  current_inst = instances.end();

  updateMarker();
}

int top10::tracked::MeshInstanceEditor::loadXml( const TiXmlElement* xml_node )
{
  int status = 0;
  for (const TiXmlElement* item = xml_node->FirstChildElement("mesh_instance"); item; item = item->NextSiblingElement("mesh_instance"))
  {
    addInstance();
    MeshInstance& new_one = instances.back();
    
    int s;
    int id = -1;
    s = item->QueryIntAttribute("mesh_id", &id);
    if (s < 0) {
      logXmlNodeError("Could not get the id of the mesh", item, Log::Error);
      status = -1;
    }
    else new_one.mesh_id = id;
    setMeshNode();

    double val;
    s = item->QueryDoubleAttribute("translation_x", &val);
    if (s < 0) {
      logXmlNodeError("Could not get the x translation of the mesh", item, Log::Error);
      status = -1;
    }
    else new_one.translation.x = val;

    s = item->QueryDoubleAttribute("translation_y", &val);
    if (s < 0) {
      logXmlNodeError("Could not get the y translation of the mesh", item, Log::Error);
      status = -1;
    }
    else new_one.translation.y = val;

    s = item->QueryDoubleAttribute("translation_z", &val);
    if (s < 0) {
      logXmlNodeError("Could not get the z translation of the mesh", item, Log::Error);
      status = -1;
    }
    else new_one.translation.z = val;

    s = item->QueryDoubleAttribute("rotation", &val);
    if (s < 0) {
      logXmlNodeError("Could not get the rotation of the mesh", item, Log::Error);
      status = -1;
    }
    else new_one.rotation = val;

    new_one.transform_node->setToWorld(getInstanceTransform());
  }
  
  updateMarker();

  return status;
}

int top10::tracked::MeshInstanceEditor::saveXml( TiXmlElement * xml_node ) const
{
  int status = 0;
  for (InstanceList::const_iterator it = instances.begin(); it != instances.end(); ++it)
  {
    TiXmlElement item("mesh_instance");
    item.SetAttribute("mesh_id", it->mesh_id);
    item.SetDoubleAttribute("translation_x", it->translation.x);
    item.SetDoubleAttribute("translation_y", it->translation.y);
    item.SetDoubleAttribute("translation_z", it->translation.z);
    item.SetDoubleAttribute("rotation", it->rotation);
    xml_node->InsertEndChild(item);
  }
  return status;
}

void top10::tracked::MeshInstanceEditor::setMeshNode()
{
  current_inst->transform_node->removeChild(current_inst->mesh_node.getPtr());
  current_inst->mesh_node = mesh_lib_ed->getNode(current_inst->mesh_id);
  current_inst->transform_node->addChild(current_inst->mesh_node.getPtr());

  updateMarker();
}