/*
  Top10, a racing simulator
  Copyright (C) 2000-2005  Johann Deneux
  
  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.
  
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
  
  Authors can be contacted at following electronic addresses:
  Johann Deneux: johann.deneux@gmail.com
*/


#ifndef TOP10_TRACKED_MAINEDITOR_HH
#define TOP10_TRACKED_MAINEDITOR_HH

#include "Drawable.hh"
#include "StartingAreaEditor.hh"
#include "CheckpointsEditor.hh"
#include "PathEditor.hh"
#include "SectionsEditor.hh"
#include "LayoutEditor.hh"
#include "HullEditor.hh"
#include "TriangulationEditor.hh"
#include "MeshInstanceEditor.hh"
#include "WallEditor.hh"
#include "util/XmlRespChain.hh"
#include "track/Track.hh"

namespace top10 {
  namespace tracked {
    class MainEditor {
    public:
      //! Major version number of the file format
      static const int version_major = 1;
      //! Minor version number of the file format
      static const int version_minor = 8;
      
    public:
      /*! \param owns if true, the editors are deleted when this instance is destroyed. */
      MainEditor(bool owns = false);
      ~MainEditor();
      
      //! Named constructor that builds a main editor and its own editors
      static MainEditor* makeReadyToUse();

      inline std::string getOrigin() const { return "MainEditor"; }
      
      //! Add an editor which will be used both for loading and saving
      /*! You should not add two editors handling the same kind of xml nodes */
      void addEditor(top10::util::XmlDumpable* ed);

      //! Build a track according to the data in the editors
      top10::track::Track* buildTrack(const std::string& name);
      
      //! Load a track in the new format
      int loadXml(std::string filename);
      
      //! Save a track in the new format
      int saveXml(std::string filename);
      
      //! Reset the state of each editor
      void clearAll();
      
      //! Return true if one of the editor was changed
      bool needSave() const;
      
      //! Reset the need_save flag
      void setNeedSave(bool);
      
    private:
      //! Look for an editor of a certain type
      template <typename EditorT>
          EditorT* getEditor()
      {
        for (std::list<top10::util::XmlDumpable*>::iterator it = editors.begin(); it != editors.end(); ++it) {
          if (dynamic_cast<EditorT*>(*it)) return (reinterpret_cast<EditorT*>(*it));
        }
        return 0;
      }
          
    private:
      
      //! XML document where all data from all editors are saved
      /*!
        This document must contain one section for each editor. Additional sections may be present, but their names should
        be different from the ones used by top10's editors.
        Additional sections are allowed for extensibility purposes, 3rd-party tools may save their information in these sections.
        MainEditor will not touch them.
      */
      TiXmlDocument xml_doc;

      top10::util::XmlRespChain loaders;
      std::list<top10::util::XmlDumpable*> editors;
      
      bool owns;
    };
  };
};

#endif //MAINEDITOR_H

