/*
  Top10, a racing simulator
  Copyright (C) 2000-2005  Johann Deneux
  
  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.
  
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
  
  Authors can be contacted at following electronic addresses:
  Johann Deneux: johann.deneux@gmail.com
*/

#include "MainEditor.hh"
#include "track/Track.hh"
#include "util/Log.hh"
#include "util/strconv.hh"

namespace top10 {
namespace math {
#include "math/Triangulation-template.cpp"
}
}

using namespace top10::tracked;
using top10::util::Log;

MainEditor::MainEditor(bool owns):
  xml_doc("unnamed.xml"), owns(owns)
{
  TiXmlDeclaration decl;
  xml_doc.InsertEndChild(decl);
}

MainEditor::~MainEditor()
{
  if (owns) {
    for (std::list<top10::util::XmlDumpable*>::iterator it = editors.begin(); it != editors.end(); ++it) {
      delete *it;
      *it = 0;
    }
  }
}

MainEditor* MainEditor::makeReadyToUse()
{
  MainEditor* main_ed = new MainEditor(true /*owns*/);
  
  CheckpointsEditor*  cps_ed          = new CheckpointsEditor;
  StartingAreaEditor* sa_ed           = new StartingAreaEditor;
  PathEditor*         path_ed         = new PathEditor;
  SectionsEditor*     sections_ed     = new SectionsEditor;
  LayoutEditor*       layout_ed       = new LayoutEditor;
  HullEditor*         hull_ed         = new HullEditor;
  TriangulationEditor* tri_ed         = new TriangulationEditor;
  top10::helpers::MeshEditor* mesh_lib = new top10::helpers::MeshEditor;
  MeshInstanceEditor* mesh_instance_ed = new MeshInstanceEditor;
  WallEditor*         wall_ed         = new WallEditor;

  // Setup connections between editors
  main_ed->addEditor(cps_ed);
  main_ed->addEditor(sa_ed);
  main_ed->addEditor(path_ed);
  main_ed->addEditor(sections_ed);
  main_ed->addEditor(layout_ed);
  main_ed->addEditor(hull_ed);
  main_ed->addEditor(tri_ed);
  main_ed->addEditor(mesh_lib);
  main_ed->addEditor(mesh_instance_ed);
  main_ed->addEditor(wall_ed);

  path_ed->setSectionsEditor(sections_ed);
  sections_ed->setWaypointsEditor(path_ed);
  mesh_instance_ed->setMeshEditor(mesh_lib);
  wall_ed->setPathEditor(path_ed);

  return main_ed;
}

void MainEditor::clearAll()
{
  for (std::list<top10::util::XmlDumpable*>::iterator it = editors.begin(); it != editors.end(); ++it) {
    (*it)->clearState();
  }  
}

void MainEditor::addEditor(top10::util::XmlDumpable* ed)
{
  editors.push_back(ed);
  loaders.addHandler(ed);
}

int MainEditor::saveXml(std::string filename)
{
  // Version
  bool add_version = false;
  TiXmlElement* version_node = xml_doc.FirstChildElement("track_version");
  if (!version_node) {
    add_version = true;
    version_node = new TiXmlElement("track_version");
  }
  version_node->SetAttribute("maj", version_major);
  version_node->SetAttribute("min", version_minor);
  if (add_version) {
    xml_doc.InsertEndChild(*version_node);
    delete version_node;
  }
  
  // Save the data of each editor
  int status = 0;
  for (std::list<top10::util::XmlDumpable*>::const_iterator it = editors.begin(); it != editors.end(); ++it) {
    // Remove nodes in the document handled by this editor
    for (TiXmlElement* child = xml_doc.FirstChildElement((*it)->getNodeName()); child;
         child = xml_doc.FirstChildElement((*it)->getNodeName()))
    {
      xml_doc.RemoveChild(child);
    }
    
    // Create a new node
    TiXmlElement node("NoVal");
    int s = (*it)->save(&node);
    if (s) status = s;
    xml_doc.InsertEndChild(node);
  }
  
  bool s = xml_doc.SaveFile(filename);
  return s?status:-1;
}

int MainEditor::loadXml(std::string filename)
{
  bool success = xml_doc.LoadFile(filename);
  if (!success) {
    Log::getSingle()->send(Log::Error, getOrigin(), filename + " does not exist, or it does not appear to be an Xml file");
    return -1;
  }
  
  bool succeeded = true;
  
  // Compatibility check
  TiXmlElement* version_node = xml_doc.FirstChildElement("track_version");
  if (!version_node) {
    Log::getSingle()->send(Log::Warning, getOrigin(), "Missing track_version section");
    succeeded = false;
  }
  else {
    int ver_maj=0, ver_min=0;
    version_node->Attribute("maj", &ver_maj);
    version_node->Attribute("min", &ver_min);
    // Compare major version numbers, they must match
    if (ver_maj > version_major) {
      Log::getSingle()->send(Log::Error, getOrigin(), "This track was edited with a more recent version ("+top10::util::toString(ver_maj)
          +".X) than the current one ("+top10::util::toString(version_major)+".X)");
      return -1;
    }
    else if (ver_maj < version_major) {
      Log::getSingle()->send(Log::Error, getOrigin(), "This track was edited with am older version ("+top10::util::toString(ver_maj)
          +".X) than the current one ("+top10::util::toString(version_major)+".X)");
      return -1;
    }
    assert(ver_maj == version_major);
    // Compare minor version numbers, warn if the file's is higher
    if (ver_min > version_minor) {
      Log::getSingle()->send(Log::Warning, getOrigin(),
        "This track was edited with a more recent version ("+top10::util::toString(ver_maj)+"."+top10::util::toString(ver_min)
	+") than the current one ("+top10::util::toString(version_major)+"."+top10::util::toString(version_minor));
    }
  }

  // Reset all editors
  clearAll();
  
  // Iterate over all sub nodes, and ask the editors to handle them
  for (const TiXmlElement* el = xml_doc.FirstChildElement(); el; el = el->NextSiblingElement()) {
    if (el->Value() != std::string("track_version")) {
      int s = loaders.handle(el);
      if (s < 0) succeeded = false;
      if (s > 0) {
        Log::getSingle()->send(Log::Warning, getOrigin(), std::string("Skipped xml node ")+el->Value());
      }
    }
  }
  return succeeded?0:-1;
}

bool MainEditor::needSave() const
{
  //TODO
  return true;
}
      
void MainEditor::setNeedSave(bool b)
{
  //TODO
}

top10::track::Track* MainEditor::buildTrack(const std::string& name)
{
  SectionsEditor* sections_ed = getEditor<SectionsEditor>();
  StartingAreaEditor* sa_ed = getEditor<StartingAreaEditor>();
  CheckpointsEditor* cps_ed = getEditor<CheckpointsEditor>();
  TriangulationEditor* tri_ed = getEditor<TriangulationEditor>();
  MeshInstanceEditor* mesh_ed = getEditor<MeshInstanceEditor>();
  WallEditor* wall_ed = getEditor<WallEditor>();

  if (!sections_ed) {
    Log::getSingle()->send(Log::Critical, getOrigin(), "No section editor found");
    return 0;
  }
  
  if (!sa_ed) {
    Log::getSingle()->send(Log::Critical, getOrigin(), "No starting area editor found");
    return 0;
  }
    
  if (!cps_ed) {
    Log::getSingle()->send(Log::Critical, getOrigin(), "No checkpoint editor found");
    return 0;
  }

  if (!tri_ed) {
    Log::getSingle()->send(Log::Critical, getOrigin(), "No triangulation editor found");
    return 0;
  }

  if (!mesh_ed) {
    Log::getSingle()->send(Log::Critical, getOrigin(), "No mesh instance editor found");
    return 0;
  }

  if (!wall_ed) {
    Log::getSingle()->send(Log::Critical, getOrigin(), "No wall editor found");
    return 0;
  }

  // Triangulate the track
  top10::track::SectionGraph* sg = sections_ed->makeSectionGraph();
  sg->setTextures(tri_ed->getGrassDistance(), tri_ed->getGrassTexture(), tri_ed->getSideTexture());
  sg->setMaxAngle(tri_ed->getMaxAngle());
  sg->setMinDist(tri_ed->getMinDistance());

  top10::graphX::MaterialNode* top_node = new top10::graphX::MaterialNode;
  top_node->r = top_node->g = top_node->b = 255;
  top_node->addChild(sg->getGround());

  delete sg;
  
  // The name of the material of the road
  std::set<std::string> road_mat_name;
  road_mat_name.insert(top10::track::AreaCoord::getRoadMaterialName());
  
  // ... of the grass
  std::set<std::string> grass_mat_name;
  grass_mat_name.insert(top10::track::AreaCoord::getGrassMaterialName());
  
  // ... the rest (not done yet)
  std::set<std::string> empty;
  
  top10::track::Track* ret = new top10::track::Track(top_node, name, empty,
      road_mat_name, empty, empty, empty, grass_mat_name, empty);
  
  // Mesh instances
  std::vector<top10::tracked::MeshInstanceEditor::Item> mesh_items;
  mesh_items = mesh_ed->getItems();
  for (std::vector<top10::tracked::MeshInstanceEditor::Item>::const_iterator
      it = mesh_items.begin(); it != mesh_items.end(); ++it)
  {
    ret->addDecoration(it->m_node.getPtr(), it->m_collides);
  }

  // Walls
  ret->addDecoration(wall_ed->generate(false /*mark_current*/), true);

  if (ret) {
    ret->setStartingArea(sa_ed->starting_area);
    ret->setCheckPoints(cps_ed->checkpoints);
    ret->buildTriMap();
    ret->buildDecoOctree();
  }
  else {
    Log::getSingle()->send(Log::Critical, getOrigin(), "Failed to build the track");
    return 0;
  }

  return ret;
}
