/*
  Top 10, a racing simulator
  Copyright (C) 2003,2005  Johann Deneux
  
  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.
  
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
  
  Authors can be contacted at following electronic addresses:
  Johann Deneux: johann.deneux@it.uu.se
*/
#include "LayoutEditor.hh"
#include "graphX/TextureManager.hh"

namespace top10 {
namespace tracked {

LayoutEditor::LayoutEditor()
 : top10::tracked::Drawable("layout"),
 size_x(1000.0), size_z(1000.0), altitude(0.0), grid_unit(50.0),
 grid_r(255), grid_g(255), grid_b(255),
 panel_r(255), panel_g(255), panel_b(255)
{
  using namespace top10::graphX;
  using namespace top10::util;
  using top10::math::Matrix4;
  using top10::math::Translation4;
  using top10::math::Rotation3;
  
  // The grid
  grid_transform = new TransformNode;
  Matrix4 T = Translation4(0.0, altitude, 0.0);
  grid_transform->setToWorld(T);
  
  grid_material = new MaterialNode;
  grid_material->r = grid_r;
  grid_material->g = grid_g;
  grid_material->b = grid_b;
  grid_transform->addChild(grid_material.getPtr());
  
  grid_node = new GridNode;
  grid_node->setSizeX(size_x);
  grid_node->setSizeZ(size_z);
  grid_node->setGridUnit(grid_unit);
  grid_material->addChild(grid_node.getPtr());
  
  addChild(grid_transform.getPtr());
}

void LayoutEditor::setTexture()
{
  using namespace top10::graphX;
  using namespace top10::util;
  using top10::math::Matrix4;
  using top10::math::Translation4;
  using top10::math::Rotation3;
  using top10::math::Vector;
  
  removeChild(panel_transform.getPtr());
  
  if (!filename.empty()) {
    panel_transform = new TransformNode;
    Matrix4 T = Translation4(0.0, altitude-0.1, 0.0) * Matrix4( Vector(1.0, 0.0, 0.0), Vector(0.0, 0.0, -1.0), Vector(0.0, 1.0, 0.0) );
    panel_transform->setToWorld(T);
    
    panel_material = new MaterialNode;
    panel_material->r = panel_r;
    panel_material->g = panel_g;
    panel_material->b = panel_b;
    panel_transform->addChild(panel_material.getPtr());
    
    panel_texture = new TextureNode;
    panel_texture->setTextureId(TextureManager::getInstance()->getTexture(filename));
    panel_material->addChild(panel_texture.getPtr());
    
    panel_node = new PanelNode;
    panel_node->setLowerLeft(-size_x/2.0, -size_z/2.0);
    panel_node->setUpperRight(size_x/2.0, size_z/2.0);
    panel_node->setDist(0.0);
    panel_texture->addChild(panel_node.getPtr());
    
    addChild(panel_transform.getPtr());
  }
}

float LayoutEditor::getSizeX() const
{
  return size_x;
}

void LayoutEditor::setSizeX(float theValue)
{
  size_x = theValue;
  grid_node->setSizeX(theValue);
  setTexture();
}

float LayoutEditor::getSizeZ() const
{
  return size_z;
}

void LayoutEditor::setSizeZ(float theValue)
{
  size_z = theValue;
  grid_node->setSizeZ(theValue);
  setTexture();
}

float LayoutEditor::getAltitude() const
{
  return altitude;
}

void LayoutEditor::setAltitude(float theValue)
{
  using top10::math::Translation4;
  
  altitude = theValue;
  grid_transform->setToWorld(Translation4(0.0, theValue, 0.0));
  setTexture();
}

void LayoutEditor::setFilename(std::string s)
{
  filename = s;
  setTexture();
}

std::string LayoutEditor::getFilename() const
{
  return filename;
}

void LayoutEditor::setGridUnit(float x)
{
  grid_unit = x;
  grid_node->setGridUnit(x);
}

float LayoutEditor::getGridUnit() const
{
  return grid_unit;
}
  
std::string LayoutEditor::getOrigin() const
{
  return "Layout editor";
}

int LayoutEditor::saveXml(TiXmlElement* node) const
{
  node->SetAttribute("filename", filename);
  node->SetDoubleAttribute("size_x", size_x);
  node->SetDoubleAttribute("size_z", size_z);
  node->SetDoubleAttribute("altitude", altitude);
  node->SetDoubleAttribute("unit", grid_unit);
  return 0;   
}

int LayoutEditor::loadXml(const TiXmlElement* node)
{
  const char* attr = node->Attribute("filename");
  if (attr) filename = attr;
  node->QueryFloatAttribute("size_x", &size_x);
  node->QueryFloatAttribute("size_z", &size_z);
  node->QueryFloatAttribute("altitude", &altitude);
  node->QueryFloatAttribute("unit", &grid_unit);
  
  // Update all graphX::Nodes
  setSizeX(size_x);
  setSizeZ(size_z);
  setGridUnit(grid_unit);
  setAltitude(altitude);
  
  return 0;
}

void LayoutEditor::clearState( )
{
  setSizeX(1000.0);
  setSizeZ(1000.0);
  setGridUnit(50.0);
  filename = "";
  setTexture();
}

}
}


