/*
  Top10, a racing simulator
  Copyright (C) 2000-2006  Johann Deneux
  
  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.
  
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
  
  Authors can be contacted at following electronic addresses:
  Johann Deneux: johann.deneux@gmail.com
*/

#include "KerbTemplateEditor.hh"
#include "util/findpred.hh"
#include "util/NameFinder.hh"

using namespace top10::tracked;

KerbTemplateEditor::KerbTemplateEditor(): top10::util::XmlDumpable("kerb_templates")
{
  clearState();
}

KerbTemplateEditor::~KerbTemplateEditor()
{
}

void KerbTemplateEditor::clearState()
{
  kerbs.clear();
  current = kerbs.end();
}

int KerbTemplateEditor::loadXml(const TiXmlElement* node)
{
  assert(node);

  int ret = 0;

  const TiXmlElement* child = node->FirstChildElement(KerbTemplate().getNodeName());
  while (child)
  {
    KerbTemplate new_one;
    int s = new_one.load(child);
    if (s)
      ret = s;

    const char* name_ptr = child->Attribute("name");
    std::string new_name;
    if (name_ptr)
    {
      new_name = name_ptr;
      kerbs[new_name] = new_one;
    }
    else
      logXmlNodeError("No name attribute or wrong type", child, top10::util::Log::Error);

     child = child->NextSiblingElement();
  }

  return ret;
}

int KerbTemplateEditor::saveXml(TiXmlElement* node) const
{
  assert(node);

  int ret = 0;

  node->Clear();

  for (KerbTemplates::const_iterator it = kerbs.begin(); it != kerbs.end(); ++it)
  {
    TiXmlElement new_one(it->second.getNodeName());
    int s = it->second.save(&new_one);
    if (s)
      ret = s;
    new_one.SetAttribute("name", it->first);
    node->InsertEndChild(new_one);
  }

  return ret;
}

void KerbTemplateEditor::newKerb()
{
  KerbTemplate new_one;
  // Find a unused name
  std::string new_name = top10::util::NameFinder< top10::util::Find1Named >::getNewName(kerbs.begin(), kerbs.end());
  kerbs[new_name] = new_one;
  current = kerbs.find(new_name);
}

void KerbTemplateEditor::renameKerb(const std::string& new_name)
{
  if (current == kerbs.end())
    return;

  kerbs[new_name] = current->second;
  kerbs.erase(current);
  current = kerbs.find(new_name);
}

void KerbTemplateEditor::gotoKerb(const std::string& name)
{
  current = kerbs.find(name);
}

KerbTemplate KerbTemplateEditor::getCurrent() const
{
  if (current != kerbs.end()) return current->second;

  return KerbTemplate();
}

void KerbTemplateEditor::setWidthInner(double d)
{
  if (current == kerbs.end()) return;
  current->second.width_inner = d;
}

void KerbTemplateEditor::setWidthPlateau(double d)
{
  if (current == kerbs.end()) return;
  current->second.width_plateau = d;
}

void KerbTemplateEditor::setWidthOuter(double d)
{
  if (current == kerbs.end()) return;
  current->second.width_outer = d;
}

void KerbTemplateEditor::setHeightInner(double d)
{
  if (current == kerbs.end()) return;
  current->second.height_inner = d;
}

void KerbTemplateEditor::setHeightOuter(double d)
{
  if (current == kerbs.end()) return;
  current->second.height_outer = d;
}
