/*
  Top10, a racing simulator
  Copyright (C) 2000-2006  Johann Deneux
  
  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.
  
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
  
  Authors can be contacted at following electronic addresses:
  Johann Deneux: johann.deneux@gmail.com
*/

#include "KerbInstanceEditor.hh"
#include "SectionsEditor.hh"

namespace top10
{
namespace tracked
{

KerbInstanceEditor::KerbInstanceEditor(): Drawable("kerb_instances")
{
  clearState();

  dark = new top10::graphX::MaterialNode;
  dark->r = dark->g = dark->b = 180;

  bright = new top10::graphX::MaterialNode;
  bright->r = bright->g = bright->b = 255;
}

KerbInstanceEditor::~KerbInstanceEditor()
{
}

void KerbInstanceEditor::setSectionsEditor(SectionsEditor* ed)
{
  sections_ed = ed;
}

void KerbInstanceEditor::clearState()
{
  kerbs.clear();
  current = kerbs.end();
}

int KerbInstanceEditor::loadXml(const TiXmlElement* el)
{
  assert(el);
  int ret = 0;

  std::list<KerbInstance> tmp;
  KerbInstance new_one;
  const TiXmlElement* child = el->FirstChildElement(new_one.getNodeName());
  while (child)
  {
    int s = new_one.loadXml(child);
    if (s)
    {
      ret = s;
    }
    tmp.push_back(new_one);
    child = child->NextSiblingElement(new_one.getNodeName());
  }

  kerbs.swap(tmp);
  current = kerbs.end();

  updateView();

  return ret;
}

int KerbInstanceEditor::saveXml(TiXmlElement* el) const
{
  assert(el);
  el->Clear();

  for (std::list<KerbInstance>::const_iterator it = kerbs.begin(); it != kerbs.end(); ++it)
  {
    TiXmlElement child(it->getNodeName());
    it->saveXml(&child);
    el->InsertEndChild(child);
  }

  return 0;
}

void KerbInstanceEditor::addWaypoint(int id, int pos)
{
  if (current != kerbs.end())
    current->addWaypoint(id, pos);

  updateViewCurrent();
}

void KerbInstanceEditor::removeWaypointPos(int pos)
{
  if (current != kerbs.end())
    current->removeWaypointPos(pos);

  updateViewCurrent();
}

void KerbInstanceEditor::setAbsStart(double d)
{
  if (current != kerbs.end())
    current->setAbsStart(d);

  updateViewCurrent();
}

void KerbInstanceEditor::setAbsEnd(double d)
{
  if (current != kerbs.end())
    current->setAbsEnd(d);

  updateViewCurrent();
}

void KerbInstanceEditor::setKerbTemplate(const std::string& s)
{
  if (current != kerbs.end())
    current->setType(s);

  updateViewCurrent();
}

void KerbInstanceEditor::newKerbInstance()
{
  KerbInstance new_one;
  kerbs.push_back(new_one);
  current = kerbs.end();
  --current;

  updateView();
}

void KerbInstanceEditor::gotoPos(int pos)
{
  if (pos < 0 || pos >= kerbs.size()) return;

  current = kerbs.begin();
  for (int i=0; i<pos && current != kerbs.end(); ++i)
    ++current;

  updateView();
}

KerbInstance KerbInstanceEditor::getCurrent() const
{
  if (current != kerbs.end())
    return *current;

  else return KerbInstance();
}

void KerbInstanceEditor::updateViewCurrent()
{
  bright->removeChild(bright->getChild(0));
  if (current == kerbs.end()) return;
  top10::graphX::Node* current_node = makeNode(*current);
  bright->addChild(current_node);
}

void KerbInstanceEditor::updateView()
{
  updateViewCurrent();

  top10::graphX::Node* child = dark->getChild(0);
  while (child) 
  {
    dark->removeChild(child);
    child = dark->getChild(0);
  }

  for (std::list<KerbInstance>::const_iterator it = kerbs.begin(); it != kerbs.end(); ++it)
  {
    dark->addChild(makeNode(*it));
  }
}

top10::graphX::Node* KerbInstanceEditor::makeNode(const KerbInstance& kerb)
{
  // Look for the sections connecting the waypoints
  /* -> Requires a new method in SectionsEditor?
        Or should we use the SectionGraph?
	 Depends on whether we allow for kerbs at intersections, and if yes how to handle these.
  */

  // Generate the central curve

  // Generate inner curve

  // Generate the plateau inner curve

  // Generate the plateau outer curve

  // Generate the outer curve

  // Remove self-loops from all curves
  /* -> That may cause problems with texturing, even more visible than the same problem for roads in turns.
         Perhaps the outline of the road should be generated using separate Hermite curves. That's a non-trivial math problem.
  */

  // Triangulate the kerb
  /* -> Using the TrackTriangulation code (which should be renamed?) or "directly" */

  return 0;
}

}
}
