/*
  Top10, a racing simulator
  Copyright (C) 2000-2006  Johann Deneux
  
  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.
  
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
  
  Authors can be contacted at following electronic addresses:
  Johann Deneux: johann.deneux@gmail.com
*/

#include "KerbInstance.hh"

namespace top10
{
namespace tracked
{

  KerbInstance::KerbInstance(): top10::util::XmlDumpable("kerb_instance")
  {
    clearState();
  }

  KerbInstance::~KerbInstance()
  {
  }

  void KerbInstance::clearState()
  {
    kerb_type = "";
    wp_ids.clear();
    t_start = 0.5;
    t_end = 0.5;
    side = Right;
  }

  int KerbInstance::loadXml(const TiXmlElement* el)
  {
    assert(el);

    KerbInstance tmp;
    int ret = 0;
    
    const char* pstr = el->Attribute("template");
    if (pstr) tmp.kerb_type = pstr;
    else logXmlNodeError("Missing template attribute", el, top10::util::Log::Error);

    int s;
    s = el->QueryDoubleAttribute("t_start", &tmp.t_start);
    if (s)
    {
      ret = s;
      logXmlNodeError("Missing t_start attribute or wrong type", el, top10::util::Log::Warning);
    }

    s = el->QueryDoubleAttribute("t_end", &tmp.t_end);
    if (s)
    {
      ret = s;
      logXmlNodeError("Missing t_end attribute or wrong type", el, top10::util::Log::Warning);
    }

    pstr = el->Attribute("side");
    if (pstr)
    {
      std::string str = pstr;
      if (str == "R") tmp.side = Right;
      else if (str == "L") tmp.side = Left;
      else logXmlNodeError("Expected L or R, got "+str, el, top10::util::Log::Error);
    }
    else logXmlNodeError("Missing side attribute", el, top10::util::Log::Warning);

    const TiXmlElement* child = el->FirstChildElement("wp");
    while (child)
    {
      int val;
      s = child->QueryIntAttribute("id", &val);
      if (s)
      {
	ret = s;
	logXmlNodeError("Missing id attribute or wrong type", child, top10::util::Log::Warning);
      }
      else tmp.wp_ids.push_back(val);
    }

    *this = tmp;
    return ret;
  }

  int KerbInstance::saveXml(TiXmlElement* el) const
  {
    assert(el);
    el->Clear();

    el->SetAttribute("template", kerb_type);
    el->SetDoubleAttribute("t_start", t_start);
    el->SetDoubleAttribute("t_end", t_end);
    switch (side)
    {
    case Right:
      el->SetAttribute("side", "R");
    case Left:
      el->SetAttribute("side", "L");
    }

    for (std::vector<int>::const_iterator it = wp_ids.begin(); it != wp_ids.end(); ++it)
    {
      TiXmlElement wp("wp");
      wp.SetAttribute("id", *it);
      el->InsertEndChild(wp);
    }

    return 0;
  }

  std::vector<int> KerbInstance::getWaypointsIds() const
  {
    return wp_ids;
  }

  double KerbInstance::getAbsStart() const
  {
    return t_start;
  }

  double KerbInstance::getAbsEnd() const
  {
    return t_end;
  }

  KerbInstance::Side KerbInstance::getSide() const
  {
    return side;
  }

  void KerbInstance::addWaypoint(int id, int pos)
  {
    assert(pos >= 0 && pos <= wp_ids.size());
    wp_ids.insert(wp_ids.begin()+pos, id);
  }

  void KerbInstance::removeWaypointPos(int pos)
  {
    assert(pos >= 0 && pos < wp_ids.size());
    wp_ids.erase(wp_ids.begin()+pos);
  }

  void KerbInstance::setAbsStart(double s)
  {
    t_start = s;
  }

  void KerbInstance::setAbsEnd(double s)
  {
    t_end = s;
  }

  void KerbInstance::setType(const std::string& str)
  {
    kerb_type = str;
  }

}
}