/*
  Top 10, a racing simulator
  Copyright (C) 2003,2005  Johann Deneux
  
  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.
  
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
  
  Authors can be contacted at following electronic addresses:
  Johann Deneux: johann.deneux@it.uu.se
*/
#include "HullEditor.hh"
#include "util/findpred.hh"
#include "util/strconv.hh"

using top10::util::Log;

namespace top10 {
namespace tracked {

HullEditor::HullEditor(): Drawable("hulls"), waypoints_ed(0)
{
  current_hull = hulls.end();
}

void HullEditor::setWaypointEditor(PathEditor* ed)
{
  waypoints_ed = ed;
}

void HullEditor::addHull()
{
  NamedHull new_one;
  
  // Find a unused name
  bool exists = true;
  int i=0;
  std::string name;
  while (exists) {
    name = "Unnamed"+top10::util::toString(i);
    top10::util::FindNamed finder(name);
    exists = (std::find_if(hulls.begin(), hulls.end(), finder) != hulls.end());
    ++i;
  }
  new_one.name = name;
  
  // Init other fields
  new_one.current_vertex = -1;
  new_one.invert = false;
  
  // Add to the existing hulls
  hulls.push_front(new_one);
  
  // Point to the newly added hull
  current_hull = hulls.begin();
}

void HullEditor::removeHull()
{
  // Check there is a current hull
  if (current_hull == hulls.end()) {
    Log::getSingle()->send(Log::Error, getOrigin(), "Please select a hull first.");
    return;
  }
  
  // Prepare to point to the hull after the current one
  std::list<NamedHull>::iterator after_erase = current_hull;
  ++after_erase;
  
  // Erase the current hull and point to the next one
  hulls.erase(current_hull);
  current_hull = after_erase;
}

void HullEditor::renameHull(const std::string& new_name)
{
  // Check there is a current hull
  if (current_hull == hulls.end()) {
    Log::getSingle()->send(Log::Error, getOrigin(), "Please select a hull first.");
    return;
  }
  
  // Empty new_name?
  if (new_name.empty()) {
    Log::getSingle()->send(Log::Error, getOrigin(), "The name of the hull cannot be empty.");
    return;
  }
  
  // Look in [hulls.begin -> current_hull)
  top10::util::FindNamed finder(new_name);
  if (std::find_if(hulls.begin(), current_hull, finder) != current_hull) {
    Log::getSingle()->send(Log::Error, getOrigin(), "There is already a hull with the name '"+new_name+"'");
    return;
  }
  // Look in (current_hull -> hulls.end)
  std::list<NamedHull>::iterator after = current_hull;
  ++after;
  if (std::find_if(after, hulls.end(), finder) != hulls.end()) {
    Log::getSingle()->send(Log::Error, getOrigin(), "There is already a hull with the name "+new_name+"'");
    return;
  }
  
  // Do the renaming
  current_hull->name = new_name;
}

void HullEditor::addVertex()
{
  assert(waypoints_ed);
  
  // Check there is a current hull
  if (current_hull == hulls.end()) {
    Log::getSingle()->send(Log::Error, getOrigin(), "Please select a hull first.");
    return;
  }
  
  assert(current_hull->current_vertex < (int)current_hull->hull.size());
  assert(current_hull->current_vertex >= -1);
  
  int idx = waypoints_ed->findNamed(waypoints_ed->getCurrentName());
  if (idx == -1) {
    Log::getSingle()->send(Log::Error, getOrigin(), "Please select a waypoint in the waypoint dialog first.");
    return;
  }
  
  current_hull->hull.insert(current_hull->hull.begin() + current_hull->current_vertex +1,
                            waypoints_ed->waypoints.at(idx).id);
  ++current_hull->current_vertex;
}

void HullEditor::removeVertex()
{
  // Check there is a current hull
  if (current_hull == hulls.end()) {
    Log::getSingle()->send(Log::Error, getOrigin(), "Please select a hull first.");
    return;
  }
  
  assert(current_hull->current_vertex < (int)current_hull->hull.size());
  assert(current_hull->current_vertex >= -1);
  
  // Is there are current vertex?
  if (current_hull->current_vertex == -1) {
    Log::getSingle()->send(Log::Error, getOrigin(), "Please select a vertex first.");
    return;
  }
  
  // Erase the vertex
  current_hull->hull.erase(current_hull->hull.begin() + current_hull->current_vertex);
  
  if (current_hull->current_vertex >= (int)current_hull->hull.size())
    current_hull->current_vertex = (int)current_hull->hull.size()-1;
}

void HullEditor::setCurrentVertex(int curr_v)
{
  // Check there is a current hull
  if (current_hull == hulls.end()) {
    Log::getSingle()->send(Log::Error, getOrigin(), "Please select a hull first.");
    return;
  }

  if (curr_v < -1 || curr_v >= (int)current_hull->hull.size()) {
    Log::getSingle()->send(Log::Critical, getOrigin(), "Invalid new current_vertex "+top10::util::toString(curr_v));
    return;
  }
  
  current_hull->current_vertex = curr_v;
}

void HullEditor::setInvert(bool b)
{
  // Check there is a current hull
  if (current_hull == hulls.end()) {
    Log::getSingle()->send(Log::Error, getOrigin(), "Please select a hull first.");
    return;
  }
  
  current_hull->invert = b;
}

std::string HullEditor::getCurrentHullName() const
{
  if (current_hull == hulls.end()) return "";
  else return current_hull->name;
}

std::vector<std::string> HullEditor::getHullNames() const
{
  std::vector<std::string> ret;
  for (std::list<NamedHull>::const_iterator it = hulls.begin();
       it != hulls.end();
       ++it)
  {
    ret.push_back(it->name);
  }
  return ret;
}

std::vector<std::string> HullEditor::getCurrentVertexNames() const
{
  std::vector<std::string> ret;
  if (hulls.end() == current_hull) return ret;

  assert(waypoints_ed);
  for (Hull::const_iterator it = current_hull->hull.begin();
       it != current_hull->hull.end();
       ++it)
  {
    int idx = waypoints_ed->findById(*it);
    if (idx == -1) ret.push_back(std::string("<Not found: ")+top10::util::toString(*it)+">");
    else ret.push_back(waypoints_ed->waypoints.at(idx).name);
  }
  
  return ret;
}

bool HullEditor::getInvert() const
{
  if (hulls.end() == current_hull) return false;
  return current_hull->invert;
}

int HullEditor::getCurrentVertexIdx() const
{
  if (hulls.end() == current_hull) return -1;
  else return current_hull->current_vertex;
}

/*
 * Implementation of Drawable
 */

std::string HullEditor::getOrigin() const
{
  return "HullEditor";
}

int HullEditor::loadXml(const TiXmlElement* xml_node)
{
  assert(xml_node);
  
  std::list<NamedHull> backup;
  hulls.swap(backup);
  
  try {
    // Iterate through the children
    const TiXmlElement* named_hull_node = xml_node->FirstChildElement("hull");
    while (named_hull_node) {
      try {
        loadNamedHull(named_hull_node);
      }
      catch(const std::string& err) {
        Log::getSingle()->send(Log::Warning, getOrigin()+"/XmlParser", err +
          " line "+top10::util::toString(named_hull_node->Row())+
          " column "+top10::util::toString(named_hull_node->Column()));
      }
      named_hull_node = named_hull_node->NextSiblingElement("hull");
    }
    current_hull = hulls.begin();
  }
  catch(...) {
    hulls.swap(backup);
    return -1;
  }
  
  return 0;
}

void HullEditor::loadNamedHull(const TiXmlElement* el)
{
  NamedHull new_one;
  int status;
  int value;
  
  // Name
  const char* attr = el->Attribute("name");
  if (!attr) throw std::string("Missing attribute 'name'");
  new_one.name = attr;
  
  // Invert
  status = el->QueryIntAttribute("invert", &value);
  if (status == TIXML_WRONG_TYPE) throw std::string("Wrong type, expected integer");
  else if (status == TIXML_NO_ATTRIBUTE) throw std::string("Missing attribute 'invert'");
  new_one.invert = (value != 0);
  
  // Hull
  const TiXmlElement* wp_id = el->FirstChildElement("waypoint");
  while(wp_id) {
    status = wp_id->QueryIntAttribute("ref", &value);
    if (status == TIXML_WRONG_TYPE) throw std::string("Wrong type, expected integer");
    else if (status == TIXML_NO_ATTRIBUTE) throw std::string("Missing attribute 'ref'");
    
    new_one.hull.push_back(value);
    wp_id = wp_id->NextSiblingElement("waypoint");
  }
  
  hulls.push_back(new_one);
}

int HullEditor::saveXml(TiXmlElement* node) const
{
  assert(node);
  
  node->Clear();
  node->SetValue("hulls");
  
  for (std::list<NamedHull>::const_iterator hull_it = hulls.begin();
       hull_it != hulls.end();
       ++hull_it)
  {
    TiXmlElement new_one("hull");
    new_one.SetAttribute("name", hull_it->name);
    new_one.SetAttribute("invert", hull_it->invert?1:0);
    
    for (Hull::const_iterator it = hull_it->hull.begin();
         it != hull_it->hull.end();
         ++it)
    {
      TiXmlElement new_v("waypoint");
      new_v.SetAttribute("ref", *it);
      new_one.InsertEndChild(new_v);
    }
    
    node->InsertEndChild(new_one);
  }
  
  return 0;
}

void HullEditor::clearState( )
{
  hulls.clear();
  current_hull = hulls.end();
}

}
}

