/*
  Top10, a racing simulator
  Copyright (C) 2000-2005  Johann Deneux
  
  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.
  
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
  
  Authors can be contacted at following electronic addresses:
  Johann Deneux: johann.deneux@it.uu.se
*/

#include "CheckpointsEditor.hh"
#include "util/Log.hh"
#include "util/strconv.hh"

#include <cassert>

using namespace top10::tracked;
using top10::util::Log;

CheckpointsEditor::CheckpointsEditor():
  Drawable("checkpoints"),
  top_node(0),
  camera(0)
{
  top_node = new Node(this);
  addChild(top_node.getPtr());
  current_cp = -1;
}

void CheckpointsEditor::clearState()
{
  checkpoints.clear();
  current_cp = -1;
}

int CheckpointsEditor::loadXml(const TiXmlElement* xml_node)
{
  assert(xml_node);
  
  if (xml_node->Value() != std::string("checkpoints")) {
    Log::getSingle()->send(Log::Error, getOrigin(), std::string("Expected checkpoints, got ")+xml_node->Value()+
      " on line "+top10::util::toString(xml_node->Row())+
      " column "+top10::util::toString(xml_node->Column()));
    return -1;
  }

  top10::racing::CheckPoints tmp_checkpoints;
  const TiXmlElement* cp_el = xml_node->FirstChildElement("checkpoint");
  while (cp_el) {
    try {
      top10::math::Vector p1, p2;
      const TiXmlElement *vec_el = cp_el->FirstChildElement("point1");
      if (!vec_el) throw std::string("Missing point1");
      p1.loadXml(vec_el);
      
      vec_el = cp_el->FirstChildElement("point2");
      if (!vec_el) throw std::string("Missing point2");
      p2.loadXml(vec_el);
      
      tmp_checkpoints.push_back(top10::racing::CheckPoint(p1, p2));
    }
    catch (const std::string& err) {
      Log::getSingle()->send(Log::Warning, getOrigin()+"/XmlParser", err +
          " line "+top10::util::toString(cp_el->Row())+
          " column "+top10::util::toString(cp_el->Column()));
    }
    cp_el = cp_el->NextSiblingElement("waypoint");
  }
  
  checkpoints.swap(tmp_checkpoints);
  
  return 0;
}

int CheckpointsEditor::saveXml(TiXmlElement* xml_node) const
{
  using top10::racing::CheckPoints;
  
  assert(xml_node);
  
  xml_node->SetValue("checkpoints");
  xml_node->Clear();
  
  for (CheckPoints::const_iterator cp_it = checkpoints.begin(); cp_it != checkpoints.end(); ++cp_it) {
    TiXmlElement cp_el("checkpoint");
    TiXmlElement p1_el("point1");
    TiXmlElement p2_el("point2");
    cp_it->getPoint1().saveXml(&p1_el);
    cp_it->getPoint2().saveXml(&p2_el);
    cp_el.InsertEndChild(p1_el);
    cp_el.InsertEndChild(p2_el);
    xml_node->InsertEndChild(cp_el);
  }
  
  return 0;
}

void CheckpointsEditor::pickLeft()
{
  assert(camera);
  assert(isCorrect(current_cp));

  if (current_cp == -1) throw std::string("You must create a new checkpoint first");
  
  top10::math::Vector p = checkpoints.at(current_cp).getPoint1();
  double dist = 0.0;
  const top10::math::Plane plane( top10::math::Vector(0.0, 1.0, 0.0), p.y );
  const top10::math::Frustum f = camera->getView();
  bool b = top10::math::intersectRay(plane, f.getCenter(), f.getDirection(), p, dist);

  if (b)
    checkpoints.at(current_cp).setPoint1(p);
}

void CheckpointsEditor::pickRight()
{
  assert(camera);
  assert(isCorrect(current_cp));

  if (current_cp == -1) throw std::string("You must create a new checkpoint first");
  
  top10::math::Vector p = checkpoints.at(current_cp).getPoint2();
  double dist = 0.0;
  const top10::math::Plane plane( top10::math::Vector(0.0, 1.0, 0.0), p.y );
  const top10::math::Frustum f = camera->getView();
  bool b = top10::math::intersectRay(plane, f.getCenter(), f.getDirection(), p, dist);

  if (b)
    checkpoints.at(current_cp).setPoint2(p);
}

void CheckpointsEditor::addNew()
{
  assert(isCorrect(current_cp));
  checkpoints.insert(checkpoints.begin()+(current_cp+1), top10::racing::CheckPoint());
}

void CheckpointsEditor::remove()
{
  assert(isCorrect(current_cp));
  if (current_cp == -1) throw std::string("Cannot remove before first checkpoint");
  checkpoints.erase(checkpoints.begin()+current_cp);
  --current_cp;
  assert(isCorrect(current_cp));
}

void CheckpointsEditor::next()
{
  assert(isCorrect(current_cp));
  if (current_cp+1 == (int)checkpoints.size()) throw std::string("Already on last checkpoint");
  else ++current_cp;
  assert(isCorrect(current_cp));
}

void CheckpointsEditor::previous()
{
  assert(isCorrect(current_cp));
  if (current_cp == -1) throw std::string("Already before first checkpoint");
  else --current_cp;
  assert(isCorrect(current_cp));
}

top10::racing::CheckPoint CheckpointsEditor::getCurrent() const
{
  assert(isCorrect(current_cp));
  if (current_cp == -1) return top10::racing::CheckPoint();
  else return checkpoints.at(current_cp);
}

void CheckpointsEditor::Node::renderGL(const top10::graphX::RenderingFeatures&,
  const top10::graphX::RenderState&,
  const top10::graphX::CameraNode&) const
{
  using top10::racing::CheckPoints;
  using top10::racing::CheckPoint;
  using top10::math::Vector;

  assert(ed->isCorrect(ed->current_cp));

  // Mark the current checkpoint
  if (ed->current_cp >= 0) {
    CheckPoint cp = ed->checkpoints.at(ed->current_cp);
    Vector p1 = cp.getPoint1();
    Vector p2 = cp.getPoint2();

    glBegin(GL_LINE_STRIP);
    glVertex3f(p1.x, p1.y, p1.z);
    glVertex3f(p1.x, p1.y + 5, p1.z);
    glVertex3f(p2.x, p2.y + 5, p2.z);
    glVertex3f(p2.x, p2.y, p2.z);
    glEnd();
    
    glBegin(GL_LINE_STRIP);
    glVertex3f(p1.x, p1.y + 5.5, p1.z);
    glVertex3f(p2.x, p2.y + 5.5, p2.z);
    glEnd();
  }

  // Draw other checkpoints
  for (CheckPoints::const_iterator cp = ed->checkpoints.begin();
       cp != ed->checkpoints.end();
       ++cp) {
    Vector p1 = cp->getPoint1();
    Vector p2 = cp->getPoint2();
      
    glBegin(GL_LINE_STRIP);
    glVertex3f(p1.x, p1.y, p1.z);
    glVertex3f(p1.x, p1.y + 5, p1.z);
    glVertex3f(p2.x, p2.y + 5, p2.z);
    glVertex3f(p2.x, p2.y, p2.z);
    glEnd();
  }
}

//! Check the checkpoint index is in range
bool CheckpointsEditor::isCorrect(int cp) const
{
  return (cp >= -1) && (cp+1 <= (int)checkpoints.size());
}
