/*
  Top 10, a racing simulator
  Copyright (C) 2003,2005  Johann Deneux
  
  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.
  
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
  
  Authors can be contacted at following electronic addresses:
  Johann Deneux: johann.deneux@it.uu.se
*/

#define TRIANGULATION_DEBUG 1

#include <SDL.h>
#include <GL/glu.h>

#include "util/PathFinder.hh"
#include "graphX/FreeCameraNode.hh"
#include "graphX/GroupNode.hh"
#include "graphX/Renderer.hh"
#include "graphX/GridNode.hh"
#include "graphX/MaterialNode.hh"
#include "trackedit/SectionsEditor.hh"
#include "trackedit/PathEditor.hh"
#include "TrackTriangulation.hh"
#include "SectionGraph.hh"

namespace top10 {
namespace math {
#include "math/Triangulation-template.cpp"
}
}

top10::tracked::PathEditor* path_ed = 0;
top10::tracked::SectionsEditor* sections_ed = 0;

/*
  SDL video + openGL init
*/
SDL_Surface* initGL(int depth, int w, int h, bool full_screen)
{
  SDL_Surface* screen;
  int bits_per_channel(5);

  SDL_GL_SetAttribute( SDL_GL_RED_SIZE, bits_per_channel );
  SDL_GL_SetAttribute( SDL_GL_GREEN_SIZE, bits_per_channel );
  SDL_GL_SetAttribute( SDL_GL_BLUE_SIZE, bits_per_channel );
  SDL_GL_SetAttribute( SDL_GL_DEPTH_SIZE, 16 );
#if 1
  SDL_GL_SetAttribute( SDL_GL_DOUBLEBUFFER, 1 );
#else
  SDL_GL_SetAttribute( SDL_GL_DOUBLEBUFFER, 0 );
#endif
  screen = SDL_SetVideoMode( w, h, depth, SDL_OPENGLBLIT | (full_screen?SDL_FULLSCREEN:0));
  if (screen == NULL) {
    throw std::string("Could not set GL mode: ") + std::string(SDL_GetError());
  }

  return screen;
}

top10::track::TrackTriangulation* load(const std::string& filename, bool inv)
{
  // Create only the required editors to load the track outline
  path_ed = new top10::tracked::PathEditor;
  sections_ed = new top10::tracked::SectionsEditor;
  sections_ed->setWaypointsEditor(path_ed);
  
  // Load the xml track file
  TiXmlDocument xml_doc;
  bool success = xml_doc.LoadFile(filename);
  if (!success) {
    throw top10::util::Error(filename + " does not exist, or it does not appear to be an Xml file");
  }
  
  // Look for the interesting xml nodes
  // waypoints
  TiXmlElement* xml_node = xml_doc.FirstChildElement("waypoints");
  if (!xml_node) {
    throw top10::util::Error("Could not find section waypoints");
  }
  int status = path_ed->loadXml(xml_node);
  if (status < 0) throw top10::util::Error("Failed to load waypoints");
  
  // sections
  xml_node = xml_doc.FirstChildElement("paths");
  if (!xml_node) {
    throw top10::util::Error("Could not find section paths");
  }
  status = sections_ed->loadXml(xml_node);
  if (status < 0) throw top10::util::Error("Failed to load track paths");
  
  // Build the outline
  return sections_ed->makeTriangulation(inv);
}

int main(int argc, char** argv)
{
  try {
    int width = 800;
    int height = 600;
    std::string filename;
    bool inv = false;
    
    // Parse comand line options
    for (int i=1; i<argc; ++i) {
      if (std::string(argv[i]) == "-w") {
        ++i;
        if (i >= argc) throw top10::util::Error("Missing value: width");
        width = atoi(argv[i]);
      }
      else if (std::string(argv[i]) == "-h") {
        ++i;
        if (i >= argc) throw top10::util::Error("Missing value: height");
        height = atoi(argv[i]);
      }
      else if (std::string(argv[i]) == "-inv") {
        inv = true;
      }
      else {
        filename = argv[i];
      }
    }
    if (filename.empty()) throw top10::util::Error("Missing filename");
    
    // Find the location of the track file
    top10::util::PathFinder path_finder = top10::util::PathFinder::defaultPathFinder();
    std::string path = path_finder.find(filename);
    if (path.empty()) {
      throw top10::util::Error("Could not find ")+filename;
    }

    // Load the track and build the triangulation, with the holes
    top10::track::TrackTriangulation* triangulation = load(path, inv);
    sections_ed->setVisible(true);
    sections_ed->updateView();
    triangulation->stepInit();
    
    // SDL init
    if ( SDL_Init(SDL_INIT_VIDEO) < 0 )
      throw std::string("Couldn't initialize SDL: ") + std::string(SDL_GetError());
    initGL(0 /* Depth not specified */, width, height, false /* No fullscreen */);

    // GL init
    top10::graphX::Renderer render;
    render.initGL(width, height);

    // Visualize triangulation
    // Create a camera    
    top10::graphX::FreeCameraNode* camera = new top10::graphX::FreeCameraNode;
    camera->setFar(10000.0);
    camera->translate(top10::math::Vector(0.0, 100.0, 0.0));
    camera->rotateX(-90.0);
    render.setCamera(camera);
    enum Action { Pick, Propagate, Done } action = Pick;
    
    // An horizontal grid
    top10::util::Ref<top10::graphX::GridNode> grid_node;
    grid_node = new top10::graphX::GridNode;
        
    for (bool leave = false; !leave;) {
      // White material
      top10::util::Ref<top10::graphX::MaterialNode> white_node;
      white_node = new top10::graphX::MaterialNode;
      white_node->r = white_node->g = white_node->b = 255;
      white_node->addChild(grid_node.getPtr());
      white_node->addChild(triangulation->makeNode());
      white_node->addChild(sections_ed->getSwitchNode());
      
      // Draw the scene
      render.clearAllGL();
      render.clearList();
      render.buildList(white_node.getPtr());
      render.renderGL();
      SDL_GL_SwapBuffers();
      
      // Handle events
      SDL_Event event;
      while (SDL_PollEvent(&event)) {
        switch (event.type) {
          case SDL_KEYDOWN:
            if (event.key.keysym.sym == SDLK_ESCAPE) leave=true;
            else if (event.key.keysym.sym == SDLK_SPACE) {
              if (action == Pick) {
                triangulation->stepPick();
                if (triangulation->stepDone()) action = Done;
                else action = Propagate;
              }
              else if (action == Propagate) {
                triangulation->stepPropagate();
                action = Pick;
              }
              else if (action == Propagate) {
                std::cout<<"Done. Press tab to restart."<<std::endl;
              }
            }
            else if (event.key.keysym.sym == SDLK_TAB) {
              triangulation->stepInit();
              action = Pick;
            }
            else if (event.key.keysym.sym == SDLK_h) {
              sections_ed->setVisible(false);
            }
            else if (event.key.keysym.sym == SDLK_s) {
              sections_ed->setVisible(true);
            }
            break;

          case SDL_MOUSEMOTION:
            if (event.motion.state == 1) {
              camera->rotateY(90.0 * event.motion.xrel / height);
              camera->rotateX(90.0 * event.motion.yrel / width);
            }
            if (event.motion.state > 1) {
              camera->translateL(top10::math::Vector(-50.0*event.motion.xrel/width, 0.0, 50.0*event.motion.yrel/height));
            }
            camera->update();

            break;
        }
      }
    }
    delete sections_ed;
    delete path_ed;
    delete triangulation;
  }
  catch (const top10::track::TrackTriangulation::SimilarPoints& e) {
    std::cerr<<e<<std::endl;
    std::cerr<<"The points were: "<<std::endl;
    std::cerr<<"pos = "<<e.v1.pos<<" s= "<<e.v1.s<<" t= "<<e.v1.t<<std::endl;
    std::cerr<<"pos = "<<e.v2.pos<<" s= "<<e.v2.s<<" t= "<<e.v2.t<<std::endl;
    
    return 1;
  }
  catch (const std::string& err) {
    std::cerr<<err<<std::endl;
    return 1;
  }
  
  return 0;
}
