/*
  Top 10, a racing simulator
  Copyright (C) 2003,2005  Johann Deneux
  
  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.
  
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
  
  Authors can be contacted at following electronic addresses:
  Johann Deneux: johann.deneux@it.uu.se
*/

#ifndef TOP10_TRACKTRACKTRIANGULATION_HH
#define TOP10_TRACKTRACKTRIANGULATION_HH

#include "math/Triangulation.hh"
#include "math/Vertex.hh"
#include "AreaCoord.hh"

namespace top10 {
  namespace graphX { class Node; }

namespace track {

//! A vertex belonging to possibly several sections or grass areas.
struct TexturedVertex
{
  TexturedVertex() {}
  TexturedVertex(double x, double y, double z): pos(x,y,z) {}
  TexturedVertex(const top10::math::Vector& pos): pos(pos) {}

  top10::math::Vector pos;
  std::vector<AreaCoord> coords;
  
  inline bool operator==(const TexturedVertex& other) const {
    return pos == other.pos && coords == other.coords;
  }

  void mergeIn(const TexturedVertex&);
};

//! Access the x,y,z components of a TexturedVertex
class TexturedVertexAccessor {
public:
  inline double getX(const TexturedVertex& v) const { return v.pos.x; }
  inline double getY(const TexturedVertex& v) const { return v.pos.y; }
  inline double getZ(const TexturedVertex& v) const { return v.pos.z; }
};

//! Used to store TexturedVertex in maps/sets
class TexturedVertexLexOrder {
public:
  inline bool operator()(const TexturedVertex& v1, const TexturedVertex& v2) const {
    return vector_order(v1.pos, v2.pos);
  }

private:
  top10::math::Vector2DLexOrder vector_order;
};

//! Used to merge TexturedVertex with identical positions, but differing AreaCoords
class TexturedVertexMerger {
public:
  inline void merge(TexturedVertex& v1, const TexturedVertex& v2) const
  {
    v1.mergeIn(v2);
  }
};

//! A triangulation of textured vertices
typedef top10::math::Triangulation<TexturedVertex, TexturedVertexAccessor, TexturedVertexMerger, TexturedVertexLexOrder> 
    TrackTriangulation;

//! Convert the result of the triangulation to something we can display
std::list<top10::graphX::Node*> convert(const TrackTriangulation::Mesh&);
}
}

#endif
