/*
  Top 10, a racing simulator
  Copyright (C) 2003,2005  Johann Deneux
  
  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.
  
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
  
  Authors can be contacted at following electronic addresses:
  Johann Deneux: johann.deneux@it.uu.se
*/
#include "TrackTriangulation.hh"
#include "graphX/MeshNode.hh"
#include "graphX/TextureTransform.hh"
#include "graphX/TextureNode.hh"
#include "graphX/TextureManager.hh"

#include <algorithm>

namespace top10 {
namespace track {

struct MeshNodeParts
{
  std::string name;
  std::string material;
  top10::math::Mesh* mesh;
  std::vector<top10::graphX::MeshNode::TextureCoord> coords;
};

std::list<top10::graphX::Node*> convert(const TrackTriangulation::Mesh& tri_mesh)
{
  // For each TextureSpec, make a MeshNode
  std::map<const TextureSpec*, MeshNodeParts> mesh_nodes;
  for (std::vector<TexturedVertex>::const_iterator it = tri_mesh.vertices.begin(); it != tri_mesh.vertices.end(); ++it)
  {
    for (std::vector<AreaCoord>::const_iterator ac = it->coords.begin(); ac != it->coords.end(); ++ac)
    {
      if (mesh_nodes.find(ac->texture) == mesh_nodes.end())
      {
	MeshNodeParts new_one;
	new_one.name = std::string("TriangulatedArea_") + top10::util::toString((int)ac->texture);
	new_one.mesh = new top10::math::Mesh;
	new_one.coords.clear();
	switch (ac->surf_type)
	{
	case Grass: new_one.material = AreaCoord::getGrassMaterialName(); break;
	case Road: new_one.material = AreaCoord::getRoadMaterialName(); break;
	//TODO: other surface types
	default: abort();
	}
	mesh_nodes[ac->texture] = new_one;
      }
    }
  }
   
  // Add vertices to all mesh_nodes
  for (std::vector<TexturedVertex>::const_iterator it = tri_mesh.vertices.begin();
       it != tri_mesh.vertices.end();
       ++it)
  {
    for (std::map<const TextureSpec*, MeshNodeParts>::iterator it2 = mesh_nodes.begin(); it2 != mesh_nodes.end(); ++it2)
    {
      it2->second.mesh->addVertex(it->pos);
    }
  }
  
  int total_clean = 0;
  // Faces
  for (std::vector<TrackTriangulation::Triangle>::const_iterator it = tri_mesh.faces.begin();
       it != tri_mesh.faces.end();
       ++it)
  {
    int added_count = 0;
    // For one of the TextureSpec which is in all lists of those 3 vertices, add a face to the right mesh
    const TexturedVertex& v1 = tri_mesh.vertices.at(it->p[0]);
    const TexturedVertex& v2 = tri_mesh.vertices.at(it->p[1]);
    const TexturedVertex& v3 = tri_mesh.vertices.at(it->p[2]);
    for (std::vector<AreaCoord>::const_iterator ac = v1.coords.begin(); added_count == 0 && ac != v1.coords.end(); ++ac)
    {
      AreaCoordFind finder(ac->texture);

      // Try to find an exact match
      std::vector<AreaCoord>::const_iterator ac2 = std::find_if(v2.coords.begin(), v2.coords.end(), finder);
      std::vector<AreaCoord>::const_iterator ac3 = std::find_if(v3.coords.begin(), v3.coords.end(), finder);

      if (ac2 != v2.coords.end() && ac3 != v3.coords.end())
      {
	mesh_nodes[ac->texture].mesh->addFace(it->p[0], it->p[1], it->p[2]);

	top10::graphX::MeshNode::TextureCoord coords;
	std::vector<top10::graphX::MeshNode::TextureCoord>& tex_coords = mesh_nodes[ac->texture].coords;
	
	coords.s = ac->s;
	coords.t = ac->t;
	tex_coords.push_back(coords);
	coords.s = ac2->s;
	coords.t = ac2->t;
	tex_coords.push_back(coords);
	coords.s = ac3->s;
	coords.t = ac3->t;
	tex_coords.push_back(coords);

	++added_count;
      }
    }

    if (added_count == 1) total_clean++;
    else if (added_count <= 0)
    {
      top10::util::Log::getSingle()->send(top10::util::Log::Warning, "TrackTriangulation/convert",
	"Failed to convert triangle, non matching texture specs");
    }
    else if (added_count > 1)
    {
      top10::util::Log::getSingle()->send(top10::util::Log::Warning, "TrackTriangulation/convert",
	"Generated "+top10::util::toString(added_count)+" identical triangles because vertices share more than one common texture spec");
    }
  }
  
  top10::util::Log::getSingle()->send(top10::util::Log::Info, "TrackTriangulation/convert",
    "Generated "+top10::util::toString(total_clean)+" clean triangles");

  std::list<top10::graphX::Node*> ret;
  for (std::map<const TextureSpec*, MeshNodeParts>::const_iterator it = mesh_nodes.begin();
    it != mesh_nodes.end();
    ++it)
  {
    top10::graphX::MeshNode* new_one = new top10::graphX::MeshNode(it->second.mesh, it->second.coords, it->second.name);
    new_one->setShadowVolumeFlag(false);
    new_one->setMaterialName(it->second.material);
    new_one->setShadowVolumeFlag(false); // Ground must not project shadows

    if (it->first && !it->first->getFilename().empty())
    {
      top10::graphX::TextureTransform* ttrans = new top10::graphX::TextureTransform;
      top10::math::Matrix4 M = top10::math::Identity4();
      M(0, 3) = it->first->getTranslateLat();
      M(1, 3) = it->first->getTranslateLong();
      M(0, 0) = it->first->getScaleLat();
      M(1, 1) = it->first->getScaleLong();
      ttrans->setToWorld(M);
      ttrans->addChild(new_one);

      top10::graphX::TextureNode* texture = new top10::graphX::TextureNode;
      texture->setFiltering(top10::graphX::RenderState::Anisotropic);
      texture->setTextureId(top10::graphX::TextureManager::getInstance()->getTexture(it->first->getFilename()));
      texture->addChild(ttrans);

      ret.push_back(texture);

      top10::util::Log::getSingle()->send(top10::util::Log::Info, "TrackTriangulation/convert",
	"Added mesh "+it->second.name+" with texture");
    }
    else
    {
      ret.push_back(new_one);
      top10::util::Log::getSingle()->send(top10::util::Log::Warning, "TrackTriangulation/convert",
	"Added mesh "+it->second.name+" without texture");
    }
  }
  
  return ret;
}

void TexturedVertex::mergeIn(const TexturedVertex& other)
{
  for (std::vector<AreaCoord>::const_iterator it = other.coords.begin(); it != other.coords.end(); ++it)
  {
    LooseAreaCoordFind finder(it->surf_type);
    if (std::find_if(coords.begin(), coords.end(), finder) == coords.end())
    {
      coords.push_back(*it);
    }
    else
    {
      top10::util::Log::getSingle()->send(top10::util::Log::Critical, "TexturedVertex::mergeIn", "Similar AreaCoord already present");
    }
  }
}

}
}
