/*
  Top 10, a racing simulator
  Copyright (C) 2003,2005  Johann Deneux
  
  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.
  
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
  
  Authors can be contacted at following electronic addresses:
  Johann Deneux: johann.deneux@it.uu.se
*/

#ifndef TOP10_TRACKSECTIONGRAPH_HH
#define TOP10_TRACKSECTIONGRAPH_HH

#include "Waypoint.hh"
#include "Section.hh"
#include "TrackTriangulation.hh"
#include "OutlineVertex.hh"
#include "util/Graph.hh"
#include "util/error.hh"
#include "graphX/GroupNode.hh"

namespace top10 {
namespace track {

  class TextureSpec;

//! Class used to create a track from its high-level description
class SectionGraph {
public:
  DECL_ERROR(OppositeExists);
  DECL_ERROR(SameExists);  
public:
  //! SectionGraph enclosed in a rectangle
  SectionGraph(double x_min, double z_min, double x_max, double z_max);
  
  //! Used by the TriangulationEditor
  void setTextures(double grass_dist, const TextureSpec* grass_texture, const TextureSpec* side_texture);

  //! Used to log messages
  static inline std::string getOrigin() { return "SectionGraph"; }
  
  //! Set the maximum angle using when vectorizing track sections
  /*! The smaller, the smoother. */
  void setMaxAngle(double degrees);
      
  //! Set the minimum distance between two vertices of the vectorized track sections.
  /*! The smaller, the smoother */
  void setMinDist(double meters);
  
  //! Add a vertex
  /*! This is only required for points which are not part of a section.
      Adding it anyway is harmless.
      This instance must be in state Creating.
  */
  void addPoint(const top10::math::Vector&);
  
  //! Add a section
  /*! Requires this instance to be in state Creating */
  void addSection(const Section&);
  
  //! Get the result of the triangulation, compute it if needed.
  top10::graphX::GroupNode* getGround();
  
  //! Usefull mostly for debugging
  inline std::list< std::vector<TexturedVertex> > getOutlines() const { return outlines; }
  
private:
  //! Get or create an id number for a waypoint
  int getIdx(const top10::math::Vector& waypoint);

private:
  enum State { Creating, Using };
  typedef std::map<top10::math::Vector, int, top10::math::VectorLexOrder> VecToInt;
  typedef std::map<int, top10::math::Vector> IntToVec;
  typedef std::map< std::pair<int, int>, Section > IdToSection;
  typedef std::list<Section> Loop;
  
  //! Depending on state, some methods are allowed/forbidden
  State state;
  //! Keep track of which id is associated to a waypoint
  VecToInt vec2int;
  //! Reverse of vec2int
  IntToVec int2vec;
  //! Keep track of sections and their ids
  IdToSection id2section;
  //! Id of next waypoint
  int next_id;
  //! Used to compute triangulations
  std::list< std::vector<TexturedVertex> > outlines;
  //! Required by triangulations
  TexturedVertexAccessor acc;
  //! Triangulation of the ground
  TrackTriangulation triang;
  //! Controls the smoothness of the vectorization 
  double vectorization_max_angle;
  //! Avoid creating too many segments in the vectorization
  double vectorization_min_dist;
  
  //! Distance between the road and the grass, a.k.a the width of the "side"
  double grass_dist;
  //! The texture of the grass
  const TextureSpec* grass_texture;
  //! The texture of the side of the road
  const TextureSpec* side_texture;

public:  
  //! Compute private data used to retrieve information
  void finishCreate();
  
private:
  //! Compute constraints related to a certain loop in the the track graph
  /*!
    \param do_side If true, generate constraints for the boundary between the side of the road and the grass, otherwise between the road and the side.
    */
  void addConstraints(int loop_n, const top10::util::Graph::Path& loop, bool do_side);

  /*!
  \param do_side If true, generate constraints for the boundary between the side of the road and the grass, otherwise between the road and the side.
  */
  bool vectorizeSection(int id1, int id2, Outline& outline, int loop_n, bool do_side);

  //! Add all waypoints which are not used in sections.
  /*!
    This waypoints are used to control the altitude of the terrain around the road.
    */
  void addHeightPoints();

  //! Check if a triangle belongs to a road segment.
  static bool isRoadTriangle(const top10::track::TexturedVertex& pt0,
                             const top10::track::TexturedVertex& pt1,
			     const top10::track::TexturedVertex& pt2);

};

}
}

#endif
