/*
  Top 10, a racing simulator
  Copyright (C) 2003,2005  Johann Deneux
  
  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.
  
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
  
  Authors can be contacted at following electronic addresses:
  Johann Deneux: johann.deneux@it.uu.se
*/
#include "SectionGraph.hh"
#include "util/Log.hh"
#include "math/Curve.hh"
#include "util/strconv.hh"
#include "TextureSpec.hh"

namespace top10 {
namespace math {
#include "math/Triangulation-template.cpp"
}
}

using top10::util::Log;
using top10::util::Graph;

namespace top10 {
namespace track {

SectionGraph::SectionGraph(double x_min, double z_min, double x_max, double z_max):
  state(Creating), next_id(0),
  triang(x_min, z_min, x_max, z_max, acc),
  vectorization_max_angle(0.2),
  vectorization_min_dist(0.25),
  grass_texture(0),
  side_texture(0),
  grass_dist(1.0)
{
}

void SectionGraph::setTextures(double dist, const TextureSpec* g_texture, const TextureSpec* s_texture)
{
  grass_dist = dist;
  grass_texture = g_texture;
  side_texture = s_texture;
}

void SectionGraph::addPoint(const top10::math::Vector& v)
{
  assert(state == Creating);
  
  getIdx(v);
}

void SectionGraph::addSection(const Section& section)
{
  assert(state == Creating);
  
  int id1, id2;
  
  // Create nodes if nessary
  id1 = getIdx(section.getPoint1());
  id2 = getIdx(section.getPoint2());
  
  // Create the edge
  IdToSection::const_iterator find_it = id2section.find( std::make_pair(id1, id2) );
  if (find_it != id2section.end()) {
    Log::getSingle()->send(Log::Error, getOrigin(), "There is already a section connecting these two points");
    throw SameExists();
  }
  find_it = id2section.find( std::make_pair(id2, id1) );
  if (find_it != id2section.end()) {
    Log::getSingle()->send(Log::Error, getOrigin(), "There is already a section connecting these rwo points in the reverse order");
    throw OppositeExists();
  }
      
  id2section[ std::make_pair(id1, id2) ] = section;
}

top10::graphX::GroupNode* SectionGraph::getGround()
{
  if (state == Creating) finishCreate();
  
  top10::graphX::GroupNode* ret = new top10::graphX::GroupNode;

  std::list<top10::graphX::Node*> nodes = convert(triang.getTriangles());
  for (std::list<top10::graphX::Node*>::iterator it = nodes.begin(); it != nodes.end(); ++it)
  {
    ret->addChild(*it);
  }

  return ret;
}

void SectionGraph::finishCreate()
{
  assert(state == Creating);
  state = Using;

  Log::getSingle()->send(Log::Debug, getOrigin()+"/finishCreate", "Started");
  
  if (!int2vec.empty()) {
    // Compute a smaller hull than the initial one
    double x_min = 1e6, z_min = 1e6, x_max  = -1e6, z_max = -1e6;
    for (IntToVec::const_iterator it = int2vec.begin(); it != int2vec.end(); ++it)
    {
      top10::math::Vector pos = it->second;
      
      if (pos.x > x_max) x_max = pos.x;
      if (pos.x < x_min) x_min = pos.x;
      if (pos.z > z_max) z_max = pos.z;
      if (pos.z < z_min) z_min = pos.z;
    }
    
    // Widen the min/max coordinates
    x_max += 100.0;
    z_max += 100.0;
    x_min -= 100.0;
    z_min -= 100.0;
    
    // Add this new outer hull to the triangulations
    TexturedVertex tv[4];
    tv[0].pos.x = x_min;
    tv[0].pos.y = 0.0;
    tv[0].pos.z = z_min;
    tv[1].pos.x = x_min;
    tv[1].pos.y = 0.0;
    tv[1].pos.z = z_max;
    tv[2].pos.x = x_max;
    tv[2].pos.y = 0.0;
    tv[2].pos.z = z_max;
    tv[3].pos.x = x_max;
    tv[3].pos.y = 0.0;
    tv[3].pos.z = z_min;

    // Set the texture coordinates
    for (int i=0; i<4; ++i)
    {
      AreaCoord ac_grass(tv[i].pos.z, tv[i].pos.x);
      ac_grass.texture = grass_texture;
      ac_grass.surf_type = Grass;
      tv[i].coords.push_back(ac_grass);
    }

    // Add constraints to triang
    for (int i=1; i<4; ++i) {
      triang.addConstraint(tv[i-1], tv[i]);
    }
    triang.addConstraint(tv[3], tv[0]);
  }
  
  // Build the graph which we will use to find the smallest loops (continuous side of roads)
  Graph graph;
  for (IdToSection::const_iterator section_it = id2section.begin(); section_it != id2section.end(); ++section_it) {
    // Compute the orientation of the section
    top10::math::Vector v = int2vec[section_it->first.second] - int2vec[section_it->first.first];
    double angle1 = atan2(v.x, v.z);
    double angle2 = angle1 + M_PI;
    if (angle2 > M_PI) angle2 -= 2*M_PI;
    
    graph.addArc(Graph::VertexId(section_it->first.first), Graph::VertexId(section_it->first.second), angle1);
    graph.addArc(Graph::VertexId(section_it->first.second), Graph::VertexId(section_it->first.first), angle2);
  }
  
  // Find the loops
  std::list<Graph::Path> loops;
  graph.getSmallestLoops(loops);
  
  Log::getSingle()->send(Log::Debug, getOrigin()+"/finishCreate", "Found "+top10::util::toString(loops.size())+" loops in the track graph");
  
  // Build the triangulation of the grass areas and the road
  int loop_n = 0;
  for (std::list<Graph::Path>::const_iterator loop_it = loops.begin(); loop_it != loops.end(); ++loop_it, ++loop_n)
  {
    addConstraints(loop_n, *loop_it, true);
    addConstraints(loop_n, *loop_it, false);
  }

  for (std::list< std::vector<TexturedVertex> >::const_iterator it = outlines.begin(); it != outlines.end(); ++it)
  {
    for (std::vector<TexturedVertex>::const_iterator v_it = it->begin(); v_it != it->end(); ++v_it)
    {
      triang.addVertex(*v_it);
    }
  }

  for (std::list< std::vector<TexturedVertex> >::const_iterator it = outlines.begin(); it != outlines.end(); ++it)
  {
    try {
      triang.addConstraints(*it);
    }
    catch (...) {}
  }

  // Add all waypoints which are not used in sections
  addHeightPoints();

  Log::getSingle()->send(Log::Debug, getOrigin()+"/finishCreate", "Done");
}

int SectionGraph::getIdx(const top10::math::Vector& wp)
{
  VecToInt::const_iterator find_it = vec2int.find(wp);
  if (find_it != vec2int.end()) return find_it->second;
  vec2int[wp] = next_id;
  int2vec[next_id++] = wp;
  return next_id-1;
}

void SectionGraph::addConstraints(int loop_n, const top10::util::Graph::Path& loop, bool do_side)
{
  if (loop.size() < 3) {
    Log::getSingle()->send(Log::Debug, getOrigin() + "/addConstraints", "loop.size() == " + top10::util::toString(loop.size()));
    return;
  }
  
  // Debug output: the loop
  std::ostringstream buf;
  buf<<"Generating constraints for hole: ";
  for (top10::util::Graph::Path::const_iterator it = loop.begin(); it != loop.end(); ++it) {
    buf<<it->id<<" ";
  }
  Log::getSingle()->send(Log::Debug, getOrigin() + "/addConstraints", buf.str());
  
  assert(loop.front() == loop.back());
  
  // Will hold the nice outline at the end
  Outline final;
  
  // Outline of the previous section
  Outline prev;

  // Outline of the current section
  Outline current;
    
  // Outline of the last section
  Outline last;
    
  Graph::Path::const_iterator wp_it = loop.end(); --wp_it; --wp_it;
  Graph::Path::const_iterator wp_it2 = wp_it; ++wp_it2;
  bool invert_dir = vectorizeSection(wp_it->id, wp_it2->id, last, loop_n, do_side);
  prev = last;
  
  wp_it = loop.begin();
  wp_it2 = wp_it; ++wp_it2;
  for ( ; wp_it != loop.end(); ++wp_it, ++wp_it2) {
    // Last iteration: wrap wp_it2 to the second waypoint (i.e. the next waypoint, since loop.front() == loop.back())
    if (wp_it2 == loop.end()) { wp_it2 = loop.begin(); ++wp_it2; }
    
    bool old_invert_dir = invert_dir;
    invert_dir = vectorizeSection(wp_it->id, wp_it2->id, current, loop_n, do_side);
    
    if (old_invert_dir != invert_dir) {
    /* If it is a joint/fork, remove the points at the end of the previous outline and
        the points at the head of the current outline so that the two outlines do not cross eachother. */
      mergeOutlines(prev, current, final);
      /* If this was the first iteration, leave final empty:
        We don't want to have the whole last section inserted here, since it might needed merging,
        and we'll do that during the last iteration. */
      if (wp_it == loop.begin()) {
	final.clear();
      }
    }
    else {
      // Add prev to final
      if (wp_it != loop.begin() && prev.size() > 1) {
	final.insert(final.end(), prev.begin()+1, prev.end());
      }
      
      // Move current to prev
      prev = current;
      current.clear();
    }
  }
  // At this point, prev contains an unmerged version of the first section
  
  // Finally build the outline itself
  removeCrossings(final);
  std::vector<TexturedVertex> outline = makeConstraints(final);
  if (!outline.empty()) {
//    triang.addConstraints(outline);
    outlines.push_back(outline);
  }
  else {
    Log::getSingle()->send(Log::Warning, getOrigin()+"/addConstraints", "Empty outline");
  }
}

bool SectionGraph::vectorizeSection(int id1, int id2, Outline& current, int loop_n, bool do_side)
{
  // Look for the section connecting the current waypoint and the next
  IdToSection::const_iterator find_direct_it = id2section.find( std::make_pair(id1, id2) );
  IdToSection::const_iterator find_invert_it = id2section.find( std::make_pair(id2, id1) );
  
  // We should have found exactly one section connecting the two waypoints
  assert(!(find_direct_it == id2section.end() && find_invert_it == id2section.end()));
  assert(!(find_direct_it != id2section.end() && find_invert_it != id2section.end()));
      
  IdToSection::const_iterator find_it;
  std::vector<top10::math::Vector> tmp_normals;
  std::vector<top10::math::Vector> tmp_center;
  std::vector<float> tmp_t_center;
  top10::math::Curve2D rd;

  // Vectorize
  if (find_invert_it != id2section.end())
  {
    find_it = find_invert_it;
    double d1 = find_it->second.getRightDist1();
    if (do_side) d1 += grass_dist;
    double d2 = find_it->second.getRightDist2();
    if (do_side) d2 += grass_dist;

    find_it->second.vectorize(d1, d2,
      vectorization_max_angle, vectorization_min_dist,
      tmp_normals, tmp_center, tmp_t_center);
    rd.addXY(tmp_t_center.front(), d1);
    rd.addXY(tmp_t_center.back(), d2);
  }
  else
  {
    find_it = find_direct_it;
    double d1 = find_it->second.getLeftDist1();
    if (do_side) d1 -= grass_dist;
    double d2 = find_it->second.getLeftDist2();
    if (do_side) d2 -= grass_dist;

    find_it->second.vectorize(d1, d2,
      vectorization_max_angle, vectorization_min_dist,
      tmp_normals, tmp_center, tmp_t_center);
    rd.addXY(tmp_t_center.front(), d1);
    rd.addXY(tmp_t_center.back(), d2);
  }

  assert(tmp_normals.size() == tmp_center.size() && tmp_center.size() == tmp_t_center.size());

  int i;
  top10::math::Vector prev_center = tmp_center.front();
  float dist = 0.0;

  // Generate vertices
  Outline outline;
  for (i=0; i<tmp_center.size(); ++i)
  {
    OutlineVertex ov;
    ov.center = tmp_center.at(i);
    ov.normal = tmp_normals.at(i);

    // The u coordinate is the distance from the entry (i.e. the last) point of the section
    dist += (ov.center - prev_center).size();
    ov.u = find_it->second.getAbsStart() + dist;

    // The v coordinate is the signed distance from the center
    ov.v = rd.getY(tmp_t_center.at(i));

    if (do_side) 
    {
      // The area coord for the grass
      AreaCoord grass_coord;
      grass_coord.s = ov.center.z + ov.normal.z;
      grass_coord.t = ov.center.x + ov.normal.x;
      grass_coord.texture = grass_texture;
      grass_coord.surf_type = Grass;
      ov.coords.push_back(grass_coord);

      // The area coord for the side
      grass_coord.s = ov.center.z + ov.normal.z;
      grass_coord.t = ov.center.x + ov.normal.x;
      grass_coord.texture = side_texture;
      grass_coord.surf_type = Grass;
      ov.coords.push_back(grass_coord);
    }
    else {
      // The AreaCoord for the road
      AreaCoord road_coord;
      road_coord.s = ov.v;
      road_coord.t = ov.u;
      road_coord.texture = find_it->second.getTextureSpec();
      road_coord.surf_type = Road;
      ov.coords.push_back(road_coord);

      // The area coord for the side
      AreaCoord grass_coord;
      grass_coord.s = ov.center.z + ov.normal.z;
      grass_coord.t = ov.center.x + ov.normal.x;
      grass_coord.texture = side_texture;
      grass_coord.surf_type = Grass;
      ov.coords.push_back(grass_coord);
    }

    outline.push_back(ov);
    prev_center = ov.center;
  }

  // Append to current
  int end_i, incr;
  if (find_it == find_invert_it)
  {
    i = (int)tmp_normals.size() -1;
    end_i = -1;
    incr = -1;
    prev_center = tmp_center.back();
  }
  else
  {
    i = 0;
    end_i = (int)tmp_normals.size();
    incr = +1;
    prev_center = tmp_center.front();
  }

  for (; i != end_i; i += incr)
  {
    if (current.empty())
      current.push_back(outline.at(i));
    else if (current.back().center != outline.at(i).center)
      current.push_back(outline.at(i));
    else
      current.back().coords.insert(current.back().coords.end(), outline.at(i).coords.begin(), outline.at(i).coords.end());
  }

  return find_it == find_invert_it;
}


void SectionGraph::setMaxAngle( double degrees )
{
  assert(Creating == state);
  vectorization_max_angle = M_PI * degrees/180.0;
}

void SectionGraph::setMinDist( double meters )
{
  assert(Creating == state);
  vectorization_min_dist = meters;
}

void SectionGraph::addHeightPoints()
{
  // For each vertex
  for (IntToVec::const_iterator it = int2vec.begin(); it != int2vec.end(); ++it)
  {
    top10::track::TexturedVertex pt;
    pt.pos = it->second;
    
    top10::track::AreaCoord grass_coord(pt.pos.z, pt.pos.x);
    grass_coord.texture = grass_texture;
    grass_coord.surf_type = top10::track::Grass;
    pt.coords.push_back(grass_coord);

    top10::track::TexturedVertex pt0, pt1, pt2;

    bool in_triang = triang.getTriangle(pt, pt0, pt1, pt2);
    if (!in_triang || !isRoadTriangle(pt0, pt1, pt2))
      triang.addVertex(pt);
  }
}

bool SectionGraph::isRoadTriangle(const top10::track::TexturedVertex& pt0,
				  const top10::track::TexturedVertex& pt1,
				  const top10::track::TexturedVertex& pt2)
{
  for (std::vector<top10::track::AreaCoord>::const_iterator it0 = pt0.coords.begin();
       it0 != pt0.coords.end();
       ++it0)
  {
    for (std::vector<top10::track::AreaCoord>::const_iterator it1 = pt1.coords.begin();
         it1 != pt1.coords.end();
         ++it1)
    {
      for (std::vector<top10::track::AreaCoord>::const_iterator it2 = pt2.coords.begin();
           it2 != pt2.coords.end();
           ++it2)
      {
	if (it0->surf_type != top10::track::Grass && 
	    it0->surf_type == it1->surf_type && it1->surf_type == it2->surf_type &&
	    it0->texture == it1->texture && it1->texture == it2->texture)
	  return true;
      }
    }
  }

  return false;
}

}
}


