/*
  Top 10, a racing simulator
  Copyright (C) 2003,2005  Johann Deneux
  
  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.
  
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
  
  Authors can be contacted at following electronic addresses:
  Johann Deneux: johann.deneux@it.uu.se
*/

#ifndef TOP10_TRACKSECTION_HH
#define TOP10_TRACKSECTION_HH

#include "math/Hermite.hh"
#include "math/Matrix.hh"
#include "TextureSpec.hh"

namespace top10 {
namespace track {

//! Describes a part of a road.
/*! A part of the track, i.e. a road, is also refered to as a "section", or a "path".
*/
class Section {
public:
  Section();

  inline void setAbsStart(double a) { abs_start = a; }
  inline double getAbsStart() const { return abs_start; }

  //! Set the curve representing the "center" of the section
  void setCurve(const top10::math::Hermite& h);
  inline top10::math::Hermite getCurve() const { return curve; }
  
  //! Extract an outline of the section
  int vectorize(double dist1, double dist2, double max_angle, double min_dist,
                std::vector<top10::math::Vector>& out,
                double t_start=0.0, double t_end=1.0) const;
  
  //! Extract an outline of the section, together with information which can be used to "clean" the outline
  int vectorize(double dist1, double dist2, double max_angle, double min_dist,
                std::vector<top10::math::Vector>& normals,
                std::vector<top10::math::Vector>& center,
                std::vector<float>& t_center,
                double t_start=0.0, double t_end=1.0) const;
  
  inline top10::math::Vector getPoint1() const { return curve.getEnd1(); }
  inline top10::math::Vector getPoint2() const { return curve.getEnd2(); }
  
  void setLeftDist1(double d);
  inline double getLeftDist1() const { return left_dist1; }
  void setRightDist1(double d);
  inline double getRightDist1() const { return right_dist1; }
  
  void setLeftDist2(double d);
  inline double getLeftDist2() const { return left_dist2; }
  void setRightDist2(double d);
  inline double getRightDist2() const { return right_dist2; }
  
  void setBankAngle1(double d);
  void setBankAngle2(double d);
  inline double getBankAngle1() const { return bank_angle1; }
  inline double getBankAngle2() const { return bank_angle2; }

  inline void setTextureSpec(const TextureSpec* s) { texture = s; }
  inline const TextureSpec* getTextureSpec() const { return texture; }
  
  ~Section();

private:
  //! Curve describing the center of the road
  /*! Actually, this curve need not in general be in the centre of the road, since
    left_dist and right_dist may be different. */
  top10::math::Hermite curve;
  
  //! Signed distance of the left side from the curve.
  /*! Should be negative. */
  double left_dist1;
  double left_dist2;
  
  //! Signed distance of the right side from the curve.
  /*! Should be positive. */
  double right_dist1;
  double right_dist2;
  
  double bank_angle1;
  double bank_angle2;
  
  //! Distance from "starting line" at the entry point
  double abs_start;

  //! Pointer to the texture specification for this road section, owned by the SectionEditor.
  const TextureSpec* texture;
};

}
}

#endif
