/*
  Top 10, a racing simulator
  Copyright (C) 2003,2005  Johann Deneux
  
  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.
  
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
  
  Authors can be contacted at following electronic addresses:
  Johann Deneux: johann.deneux@it.uu.se
*/
#include "Section.hh"
#include "math/Matrix.hh"
#include "util/Log.hh"

using top10::math::Hermite;
using top10::math::Vector;
using top10::math::Matrix3;
using top10::math::Rotation3;

namespace top10 {
namespace track {

Section::Section(): left_dist1(0.0), right_dist1(0.0), left_dist2(0.0), right_dist2(0.0),
  bank_angle1(0.0), bank_angle2(0.0), abs_start(0.0)
{
}

Section::~Section()
{
}

void Section::setCurve(const Hermite& h)
{
  curve = h;
}

void Section::setLeftDist1(double d)
{
  left_dist1 = d;
}

void Section::setRightDist1(double d)
{
  right_dist1 = d;
}

void Section::setLeftDist2(double d)
{
  left_dist2 = d;
}

void Section::setRightDist2(double d)
{
  right_dist2 = d;
}

void Section::setBankAngle1(double d)
{
  bank_angle1 = d; 
}

void Section::setBankAngle2(double d)
{
  bank_angle2 = d;
}

int Section::vectorize(double dist1, double dist2, double max_angle, double min_dist,
                       std::vector<top10::math::Vector>& out_normals,
                       std::vector<top10::math::Vector>& out_center,
                       std::vector<float>& out_t_center,
                       double t_start, double t_end) const
{
  std::vector<top10::math::Vector> normals, center;
  std::vector<float> t_center;
  
  // Vectorize the center
  curve.vectorize(max_angle, min_dist, center, t_start, t_end, &t_center);
  assert(center.size() == t_center.size());
      
  // Compute normals
  // Used to remove points with too smal normals from center
  std::vector<Vector> center2;
  std::vector<float> t_center2;
  for (int i=0; i<(int)center.size(); ++i) {
    Vector tangent = curve.eval_tg(t_center[i]);
    Vector normal = tangent ^ Vector(0.0, 1.0, 0.0);
    double bank_angle = bank_angle2*t_center[i] + bank_angle1*(1.0-t_center[i]);
    Matrix3 R = Rotation3(bank_angle, tangent);
    normal = R*normal;
    double s = normal.size();
    if (s > SMALL_VECTOR) {
      normal/=s;
      float d = dist2*t_center[i] + dist1*(1.0-t_center[i]);
      normals.push_back(d*normal);
      center2.push_back(center[i]);
      t_center2.push_back(t_center[i]);
    }
  }
  center2.swap(center);
  t_center2.swap(t_center);
  
  assert(center.size() == normals.size());
  assert(t_center.size() == normals.size());

  int n = normals.size();
  
  out_normals.insert(out_normals.end(), normals.begin(), normals.end());
  out_center.insert(out_center.end(), center.begin(), center.end());
  out_t_center.insert(out_t_center.end(), t_center.begin(), t_center.end());
  return n;
}

int Section::vectorize(double dist1, double dist2, double max_angle, double min_dist, std::vector<Vector>& out,
                      double t_start, double t_end) const
{
  std::vector<Vector> center;
  std::vector<float> t_center;
  std::vector<Vector> normals;
  vectorize(dist1, dist2, max_angle, min_dist, normals, center, t_center, t_start, t_end);
    
  // Compute the result
  int last_n = center.size()-1;
  int count = 0;
  for (int i=0; i<=last_n; ++i) {
    ++count;
    out.push_back(center[i] + normals[i]);
  }
      
  return count;
}

}
}



