/*
  Top 10, a racing simulator
  Copyright (C) 2006  Johann Deneux
  
  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.
  
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
  
  Authors can be contacted at following electronic addresses:
  Johann Deneux: johann.deneux@it.uu.se
*/

#include "OutlineVertex.hh"
#include "util/Log.hh"
#include "math/Vec2D.hh"
#include "util/Graph.hh"

using top10::util::Log;

namespace top10
{
namespace track
{

std::vector<AreaCoord> mergeCoords(const std::vector<AreaCoord>& coords1, const std::vector<AreaCoord>& coords2, double t0)
{
  return coords1; //TODO, that's just a stub
}

std::vector<TexturedVertex> makeConstraints(const Outline& outline)
{
  std::vector<TexturedVertex> ret;
  
  // This outline would not generate a proper constraint
  if (outline.size() < 2) {
    Log::getSingle()->send(Log::Debug, "makeConstraints", "normals.size() < 2");
    return ret;
  }

  Outline::const_iterator v_it=outline.begin();
  ++v_it;
  // Iterate from begin+1 to end-1
  OutlineVertex prev = outline.front();
  for (int i=1; i+1<outline.size(); ++i, ++v_it)
  {
    TexturedVertex tv(v_it->center + v_it->normal);
    tv.coords = v_it->coords;
    ret.push_back(tv);

    prev = *v_it;
  }

  // The point closing the loop
  TexturedVertex tv(outline.front().center + outline.front().normal);

  // Texture coords for the starting point
  tv.coords = outline.front().coords;

  // Texture coords for the closing point
  for (std::vector<AreaCoord>::const_iterator it = v_it->coords.begin(); it != v_it->coords.end(); ++it)
  {
    if (it->surf_type == Road)
      tv.coords.push_back(*it);
  }

  ret.push_back(tv);

  return ret;
}

void mergeOutlines(Outline& prev, Outline& current, Outline& final)
{
  /*
  Section::removeCrossings(prev_normals, prev_center, prev_t_center);
  Section::removeCrossings(current_normals, current_center, current_t_center);
  */

  // Trivial case: prev is empty
  if (prev.empty())
  {
    prev.swap(current);
    // No need for current.clear();
    return;
  }
  
  // Easy case: prev contains one element.
  /* Move prev to final, move current to prev */
  if (prev.size() == 1) {
    final.push_back(prev.front());
    prev=current;
    current.clear();
    return;
  }
  
  /* Otherwise, move elements 0..idx from prev to final, where segment (idx, idx+1) intersects
    some segment (first_curr-1, first_curr) in current.
    Then move elements firs_curr..end() from current to prev
    */

  using top10::math::Vec2D;

  // Find the intersection
  bool intersect = false;
  double inter_t;
  top10::math::Vector inter_pt;
  int first_curr = 0;
  for (int i=1; !intersect && i<(int)prev.size(); ++i)
  {
    for (int j=1; !intersect && j<(int)current.size(); ++j)
    { 
      Vec2D prev_pt(prev.at(i-1).center.x + prev.at(i-1).normal.x,
	            prev.at(i-1).center.z + prev.at(i-1).normal.z);
      Vec2D prev_dir = Vec2D(prev.at(i).center.x + prev.at(i).normal.x,
	                     prev.at(i).center.z + prev.at(i).normal.z)
                     - prev_pt;
      
      Vec2D curr_pt(current.at(j-1).center.x + current.at(j-1).normal.x,
	            current.at(j-1).center.z + current.at(j-1).normal.z);
      Vec2D curr_dir = Vec2D(current.at(j).center.x + current.at(j).normal.x,
	                     current.at(j).center.z + current.at(j).normal.z)
                     - curr_pt;

      intersect = segment_intersection(prev_pt, prev_dir, curr_pt, curr_dir, inter_t);
      if (intersect)
      {
	top10::math::Vector p1 = prev.at(i-1).center + prev.at(i-1).normal;
	top10::math::Vector p2 = prev.at(i).center + prev.at(i).normal;
	inter_pt = p1 + inter_t*(p2-p1);
      }
      first_curr = j;
    }
    
    if (!intersect)
      final.push_back(prev.at(i-1));
    else
    {
      // Add to final
      OutlineVertex new_one;
      OutlineVertex p1 = prev.at(i-1);
      OutlineVertex p2 = prev.at(i);
      double t0 = (1.0-inter_t);
      new_one.center = t0*p1.center + inter_t*p2.center;
      new_one.coords = mergeCoords(p1.coords, p2.coords, t0);
      new_one.normal = inter_pt - new_one.center;
      new_one.u = t0*p1.u + inter_t*p2.u;
      new_one.v = t0*p1.v + inter_t*p2.v;
      final.push_back(new_one);

      // Insert into current
      p1 = current.at(first_curr-1);
      p2 = current.at(first_curr);
      new_one.coords = mergeCoords(p1.coords, p2.coords, t0);
      new_one.u = t0*p1.u + inter_t*p2.u;
      new_one.v = t0*p1.v + inter_t*p2.v;
      current.insert(current.begin()+first_curr, new_one);
    }
  }
  
  if (!intersect) {
    Log::getSingle()->send(Log::Debug, "mergeOutlines", "No intersection found");
  }
  
  // Move the remaining of current to prev
  prev.clear();
  prev.insert(prev.end(), current.begin()+first_curr, current.end());
  current.clear();
}

void removeCrossings(Outline& outline)
{
#if 0
  int last_n = outline.size()-1;
  
  // A graph keeping track of what normal crosses which normals
  top10::util::Graph crossers;
  for (int i=0; i<=last_n; ++i) {
    crossers.addVertex(i);
    for (int j=i+1; j<=last_n; ++j) {
      if (cross(outline[i].center, outline[i].normal, outline[j].center, outline[j].normal))
	crossers.addEdge(i, j);
    }
  }

  // Remove (some of) the offending normals, so that they no longer intersect.
  std::set<top10::util::Graph::VertexId> exclude;
  exclude.insert(0);
  exclude.insert(last_n);
  crossers.separate(exclude);
  
  // Update outline
  std::vector<top10::util::Graph::VertexId> idxs;
  crossers.getVertices(idxs);
  Outline out2;
  
  for (std::vector<top10::util::Graph::VertexId>::const_iterator idx = idxs.begin(); idx != idxs.end(); ++idx)
    out2.push_back(outline[idx->id]);

  outline.swap(out2);
#else
  int out_size = outline.size();

  if (out_size <= 4) return;

  // Set of indices of vertices to remove
  std::vector<bool> remove(out_size, false);

  for (int i=0; i<out_size; ++i)
  {
    // The index after i
    int i2 = (i+1)%out_size;
    // Segment between i and i2
    top10::math::Vector d1 =
      (outline.at(i2).center+outline.at(i2).normal) - 
      (outline.at(i).center+outline.at(i).normal);
    top10::math::Vector p1 = outline.at(i).center+outline.at(i).normal;

    /* For each vector, check if there is an intersection
       We want to remove segments between k and l if (k, k+1)  and (l, l+1) intersect,
       with l-k <= k-l, and no segment between k and l intersect with (k, k+1) */
    int j = (i2+1)%out_size;
    int j2 = (j+1)%out_size;

    /* We only look in the next 25% of the outline. In the context where this method is used,
      that should be enough, and we avoid complications related to handling (k, k+1) and (l, l+1) twice.
      */
    int last_j = (i2 + out_size/4)%out_size;
    bool found = false;
    while (j!=last_j && !found)
    {
      top10::math::Vector d2 =
	(outline.at(j2).center+outline.at(j2).normal) - 
	(outline.at(j).center+outline.at(j).normal);
      top10::math::Vector p2 = outline.at(j).center+outline.at(j).normal;

      if (cross(p1, d1, p2, d2))
      {
	// We will remove all vertices between i and j+1
	for (int j3 = i2; j3 != j2; j3 = (j3+1)%out_size) remove.at(j3) = true;
	found = true;
      }

      j = j2;
      j2 = (j2+1)%out_size;
    }
  }

  Outline tmp;
  for (int i=0; i<out_size; ++i)
  {
    if (!remove.at(i)) tmp.push_back(outline.at(i));
  }
  outline.swap(tmp);
#endif
}

}
}
