/*
  Top10, a racing simulator
  Copyright (C) 2007  Johann Deneux
  
  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.
  
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
  
  Authors can be contacted at following electronic addresses:
  Johann Deneux: johann.deneux@gmail.com
*/

#include <iostream>
#include <string>
#include <sstream>

#include "helpers/initGL.hh"
#include "helpers/MeshFactory.hh"
#include "util/SlowTime.hh"
#include "graphX/SphereNode.hh"
#include "graphX/TexGenModeNode.hh"
#include "graphX/TextureNode.hh"
#include "graphX/CubeMap.hh"
#include "graphX/FreeCameraNode.hh"
#include "graphX/Renderer.hh"
#include "graphX/LightNode.hh"
#include "graphX/GridNode.hh"
#include "graphX/MaterialNode.hh"
#include "graphX/TextureTransform.hh"
#include "graphX/TextureUnit.hh"
#include "graphX/SkyBoxNode.hh"
#include "graphX/TextNode.hh"

#include <SDL/SDL.h>
#include <plib/fnt.h>

#define USE_SPHERE 0

// In degrees
const double span_w = 45.0;  
const double span_h = 22.0;
const double move_speed = 1.0;

int win_w = 400;
int win_h = 400;
bool fullscreen = false;
bool quit = false;

bool move_forward = false;
bool move_backward = false;
bool move_left = false;
bool move_right = false;

top10::graphX::RenderState::TexCoordGenMode gen_mode = top10::graphX::RenderState::Reflective;

enum TexTransformMode { OrientInv, Orient, None } tex_trans_mode = Orient;

bool grabbed = true;
bool grab_changed = false;

void handleKeyDown(SDL_Event* event)
{
  switch (event->key.keysym.sym)
  {
    // Move camera, FPS-style
  case SDLK_w: move_forward = true; break;
  case SDLK_s: move_backward = true; break;
  case SDLK_a: move_left = true; break;
  case SDLK_d: move_right = true; break;

    // Change texture generation mode
  case SDLK_1: gen_mode = top10::graphX::RenderState::ObjectLinear; break;
  case SDLK_2: gen_mode = top10::graphX::RenderState::EyeLinear; break;
  case SDLK_3: gen_mode = top10::graphX::RenderState::Spherical; break;
  case SDLK_4: gen_mode = top10::graphX::RenderState::Normal; break;
  case SDLK_5: gen_mode = top10::graphX::RenderState::Reflective; break;

    // Change texture transform mode
  case SDLK_8: tex_trans_mode = OrientInv; break;
  case SDLK_9: tex_trans_mode = Orient; break;
  case SDLK_0: tex_trans_mode = None; break;

    // Reset modes
  case SDLK_z: gen_mode = top10::graphX::RenderState::Reflective;
               tex_trans_mode = OrientInv;
	       break;
    
    // Grab
  case SDLK_g: grabbed = !grabbed; grab_changed = true; break;

  case SDLK_ESCAPE: quit = true; break;
  }
}

void handleKeyUp(SDL_Event* event)
{
  switch (event->key.keysym.sym)
  {
  case SDLK_w: move_forward = false; break;
  case SDLK_s: move_backward = false; break;
  case SDLK_a: move_left = false; break;
  case SDLK_d: move_right = false; break;
  }
}

void handleMouse(SDL_Event* event, top10::graphX::FreeCameraNode* camera)
{
  camera->rotateY((-event->motion.xrel * span_w) / win_w);
  camera->rotateX((-event->motion.yrel * span_h) / win_h);
}

int main(int argc, char **argv)
{
  using namespace std;

  int exit_status = 0;
  try
  {
    // SDL library init
    if ( SDL_Init(SDL_INIT_NOPARACHUTE | SDL_INIT_TIMER | SDL_INIT_VIDEO | SDL_INIT_JOYSTICK | SDL_INIT_AUDIO) < 0 ) {
      throw string("Could not initialize SDL: ") + string(SDL_GetError());
    }
    SDL_ShowCursor(0);
    SDL_WM_GrabInput(SDL_GRAB_ON);

    // OpenGL init
    SDL_Surface* surf = top10::helpers::initGL(win_w, win_h, fullscreen);
    if (!surf)
      throw string("Could not initialize OpenGL");

    // Fonts
    fntInit();
    fntTexFont font;
    font.load("data/fonts/default.txf");

#if USE_SPHERE
    // A simple sphere, with no texture
    top10::util::Ref<top10::graphX::SphereNode> sphere(new top10::graphX::SphereNode);
    top10::util::Ref<top10::graphX::MaterialNode> sphere_color(new top10::graphX::MaterialNode);
    sphere_color->r = sphere_color->g = sphere_color->b = 255;
    sphere_color->addChild(sphere.getPtr());
#else
    // A go-kart, with texture.
    top10::util::Ref<top10::graphX::Node> go_kart(top10::helpers::MeshFactory::getSingle()->load("data/kart_data/default/kart.3ds"));
#endif

    // The texture of the go-kart uses texture unit 0.
    top10::util::Ref<top10::graphX::TextureUnit> tex_unit0(new top10::graphX::TextureUnit);
    tex_unit0->setUnit(0);
#if USE_SPHERE
    tex_unit0->addChild(sphere_color.getPtr());
#else
    tex_unit0->addChild(go_kart.getPtr());
#endif

    // Generate texture coordinates automatically
    top10::util::Ref<top10::graphX::TexGenModeNode> texgen(new top10::graphX::TexGenModeNode);
    texgen->setMode(gen_mode);
    texgen->addChild(tex_unit0.getPtr());

    // Correct texture coordinates according to the camera's orientation
    top10::util::Ref<top10::graphX::TextureTransform> tex_trans(new top10::graphX::TextureTransform);
    tex_trans->addChild(texgen.getPtr());

    // Enable cube mapping
    top10::graphX::CubeMap cube_map("data/textures/Orient/front.png", "data/textures/Orient/back.png",
				    "data/textures/Orient/left.png", "data/textures/Orient/right.png",
				    "data/textures/Orient/top.png", "data/textures/Orient/bottom.png");
    top10::util::Ref<top10::graphX::TextureNode> cube_texture(new top10::graphX::TextureNode);
    cube_texture->setCubeMap(cube_map);
    cube_texture->setEnvMode(top10::graphX::RenderState::Modulate);
    cube_texture->addChild(tex_trans.getPtr());

    // Set the texture unit: use unit 1 (unit 0 is used by the model's textures)
    top10::util::Ref<top10::graphX::TextureUnit> tex_unit(new top10::graphX::TextureUnit);
#if USE_SPHERE
    tex_unit->setUnit(0);
#else
    tex_unit->setUnit(1);
#endif
    tex_unit->addChild(cube_texture.getPtr());

    // A light for the sphere or go-kart
    top10::util::Ref<top10::graphX::LightNode> sphere_light(new top10::graphX::LightNode);
    top10::graphX::Light light(top10::graphX::Light::SolarLight);
    top10::math::Vector dir(-1.0, -1.0, -1.0);
    dir /= dir.size();
    light.setDirection(dir);
    sphere_light->setLight(light);
    sphere_light->addChild(tex_unit.getPtr());

    // The top-most node
    top10::util::Ref<top10::graphX::GroupNode> top_node(new top10::graphX::GroupNode);
    top_node->addChild(sphere_light.getPtr());

    // The sky box
    top10::util::Ref<top10::graphX::SkyBoxNode> skybox(new top10::graphX::SkyBoxNode);
    skybox->setFace(top10::graphX::SkyBoxNode::Front, "data/textures/Orient/front.png");
    skybox->setFace(top10::graphX::SkyBoxNode::Right, "data/textures/Orient/right.png");
    skybox->setFace(top10::graphX::SkyBoxNode::Left, "data/textures/Orient/left.png");
    skybox->setFace(top10::graphX::SkyBoxNode::Back, "data/textures/Orient/back.png");
    skybox->setFace(top10::graphX::SkyBoxNode::Top, "data/textures/Orient/top.png");
    skybox->setFace(top10::graphX::SkyBoxNode::Bottom, "data/textures/Orient/bottom.png");

    // The HUD
    top10::util::Ref<top10::graphX::GroupNode> hud(new top10::graphX::GroupNode);
    top10::util::Ref<top10::graphX::MaterialNode> red_hud(new top10::graphX::MaterialNode);
    red_hud->r = 255;
    red_hud->g = 0;
    red_hud->b = 0;
    hud->addChild(red_hud.getPtr());

    // Frames per second
    top10::util::Ref<top10::graphX::TextNode> fps_node(new top10::graphX::TextNode);
    fps_node->setPos(-0.95, -0.95);
    fps_node->setPointSize(0.1);
    fps_node->setFont(&font);
    red_hud->addChild(fps_node.getPtr());

    // The camera
    top10::util::Ref<top10::graphX::FreeCameraNode> camera(new top10::graphX::FreeCameraNode);

    // The renderer
    top10::graphX::Renderer renderer;
    renderer.initGL(win_w, win_h);
    renderer.setCamera(camera.getPtr());

    top10::util::SlowTime timer;
    double delta_t = 0.0;
    // Event loop
    SDL_Event event;
    while (!quit)
    {
      Uint32 start_t = timer.getTicks();

      // Handle events
      while (!quit && SDL_PollEvent(&event))
      {
	switch (event.type)
	{
	case SDL_QUIT:        quit = true;          break;
	case SDL_KEYDOWN:     handleKeyDown(&event); break;
	case SDL_KEYUP:       handleKeyUp(&event);   break;
	case SDL_MOUSEMOTION: if (grabbed) { handleMouse(&event, camera.getPtr()); }  break;
	}
      }

      if (grab_changed)
      {
	grab_changed = false;
	if (grabbed)
	{
          SDL_ShowCursor(0);
	  SDL_WM_GrabInput(SDL_GRAB_ON);
	}
	else
	{
          SDL_ShowCursor(1);
	  SDL_WM_GrabInput(SDL_GRAB_OFF);
	}
      }

      // Update tex coord generation mode
      texgen->setMode(gen_mode);

      // Translate camera
      if (move_forward)
	camera->translateL(move_speed*delta_t*top10::math::Vector(0.0, 0.0, -1.0));
      if (move_backward)
	camera->translateL(move_speed*delta_t*top10::math::Vector(0.0, 0.0, 1.0));
      if (move_left)
	camera->translateL(move_speed*delta_t*top10::math::Vector(-1.0, 0.0, 0.0));
      if (move_right)
	camera->translateL(move_speed*delta_t*top10::math::Vector(1.0, 0.0, 0.0));

      // Update camera
      camera->update();

      // Update texture transform
      if (tex_trans_mode == Orient)
      {
	top10::math::Matrix4 T(camera->getView().getOrient());
	tex_trans->setToWorld(T);
      }
      else if (tex_trans_mode == OrientInv)
      {
	top10::math::Matrix4 T(camera->getView().getOrientInv());
	tex_trans->setToWorld(T);
      }
      else
      {
	top10::math::Identity4 I;
	tex_trans->setToWorld(I);
      }

      // Display FPS
      std::ostringstream buf;
      buf<<1.0/delta_t<<" fps";
      fps_node->setText(buf.str());

      renderer.clearAllGL();
      top10::graphX::RenderList skybox_rl;
      renderer.buildList(skybox.getPtr(), &skybox_rl);
      renderer.renderSkyBoxGL(skybox_rl);
      
      renderer.clearDepthGL();
      top10::graphX::RenderList main_rl;
      renderer.buildList(top_node.getPtr(), &main_rl);
      renderer.renderGL(main_rl);

      renderer.clearDepthGL();
      top10::graphX::RenderList hud_rl;
      renderer.buildList(hud.getPtr(), &hud_rl);
      renderer.renderHudGL(hud_rl);

      SDL_GL_SwapBuffers();

      delta_t = (timer.getTicks() - start_t)/1000.0;
    }
  }
  catch (const std::string& err)
  {
    cout<<"Fatal error: "<<err<<endl;
    exit_status = 1;
  }

  return exit_status;
}