/*
  Top 10, a racing simulator
  Copyright (C) 2003-2007  Johann Deneux
  
  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.
  
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
  
  Authors can be contacted at following electronic addresses:
  Johann Deneux: johann.deneux@gmail.com
*/

#include "EngineTestDlg.hh"
#include "physX/Integrator.hh"

EngineTestDlg::EngineTestDlg(QWidget* parent)
: QDialog(parent),
  m_props(new top10::physX::KartProperties),
  m_kart(m_props.getPtr()),
  m_wheel_inertia_val(1.0),
  m_time(0.0),
  m_timer_id(0)
{
  TiXmlDocument doc;
  bool status = doc.LoadFile("kart.xml");
  if (!status)
  {
    TiXmlElement save_node("kart_state");
    m_props->save(&save_node);
    TiXmlDocument doc2("kart-tmpl.xml");
    doc2.InsertEndChild(save_node);
    doc2.SaveFile();

    throw std::string("Failed to load kart.xml. Created a template file.");
  }

  m_props->load(doc.FirstChildElement(top10::physX::KartProperties::NODE_NAME));
  m_kart = top10::physX::KartState(m_props.getPtr());

  setupUi(this);
}

void EngineTestDlg::timerEvent(QTimerEvent* ev)
{
  // Disable the timer, makes debugging easier.
  killTimer(ev->timerId());

  const double next_m_time = m_time + TIMER_PERIOD/1000.0;

  // Physics update
  const double dt = UPDATE_PERIOD/1000.0;
  for (double t = m_time; t < next_m_time; t += dt)
  {
    update(t, dt);
  }
  
  m_time = next_m_time;

  // Update the LCDs
  double wheel_speed = m_kart.getClutch().getWSpeed();
  m_wheel_speed->display(wheel_speed);

  double axle_speed = m_kart.getAxle().getWSpeed();
  m_axle_speed->display(axle_speed);

  double clutch_torque = m_kart.getClutch().getFriction()*(wheel_speed - axle_speed);
  m_clutch_friction->display(clutch_torque);

  double engine_speed = m_kart.getEngine().getWSpeed();
  m_engine_speed->display(engine_speed);
  m_engine_rpm->display(engine_speed*30.0/3.14);
  m_engine_torque->display(m_kart.getEngine().getTorque());

  // Enable the timer agin.
  m_timer_id = startTimer(TIMER_PERIOD);
}

void EngineTestDlg::update(double t, double dt)
{
  using namespace top10::physX;

  m_kart.update(t, dt);
 
  double wheel_speed = m_kart.getClutch().getWSpeed();
  double axle_speed = m_kart.getAxle().getWSpeed();
  double clutch_torque = m_kart.getClutch().getFriction()*(wheel_speed - axle_speed);
  
  // Update the clutch
  DoubleState state;
  state.setPos(0.0);
  state.setSpeed(wheel_speed);

  ConstantDoubleAccel const_accel(-clutch_torque / m_wheel_inertia_val);
  integrate<DoubleState, DoubleDerivative, double, ConstantDoubleAccel>(state, m_time, TIMER_PERIOD/1000.0, const_accel);

  m_kart.m_clutch.setWSpeed(state.getSpeed());
}

void EngineTestDlg::on_m_clutch_engaged_toggled(bool b)
{
  m_kart.setClutch(b);
}

void EngineTestDlg::on_m_gear_valueChanged(int gear)
{
  m_kart.setGear(gear);
}

void EngineTestDlg::on_m_throttle_valueChanged(int percent)
{
  m_kart.setThrottle(percent/100.0);
}

void EngineTestDlg::on_m_start_stop_pressed()
{
  if (m_timer_id != 0)
  {
    killTimer(m_timer_id);
    m_timer_id = 0;
    m_start_stop->setText("Start");
  }
  else
  {
    m_timer_id = startTimer(TIMER_PERIOD);
    m_start_stop->setText("Stop");
  }
}

void EngineTestDlg::on_m_wheel_inertia_editingFinished()
{
  bool ok = false;
  double val = m_wheel_inertia->text().toDouble(&ok);
  if (ok)
    m_wheel_inertia_val = val;
}