/*
  Top10, a racing simulator
  Copyright (C) 2006  Johann Deneux
  
  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.
  
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
  
  Authors can be contacted at following electronic addresses:
  Johann Deneux: johann.deneux@gmail.com
*/
#ifndef TOP10_SOURCE_ALLOC_HH
#define TOP10_SOURCE_ALLOC_HH

#include "AudioContext.hh"
#include "math/Vertex.hh"

namespace top10
{
  namespace sound
  {
    class SourceAllocator;

    //! A wrapper around an openAL source.
    class Source: public top10::util::RefCount, private top10::util::NoCopy
    {
      friend class SourceAllocator;
    public:
      struct InvalidSource {};

      //! Return the openAL source assigned to this source, or throw InvalidSource
      /*! If this Source is invalid, an attempt is made to validate it.
          If this attempt fails, InvalidSource is thrown.
	  */
      ALuint getSource();

      //! Return true if this Source has an openAL source.
      /*! No attempt is made to validate this Source. */
      bool isValid() const;

      //! Control whether the openAL source of this Source should be looping.
      /*! If this Source is invalid, should an openAL source be assigned to this Source later,
          the openAL source will be set as looping then.
      */
      void setLooping(bool);
      
      void setGain(float);
      void setPitch(float);
      void setVelocity(const top10::math::Vector&);
      void setPosition(const top10::math::Vector&);

      //! Set the priority level.
      /*! Priority 0 means good chances of being assigned an openAL source.
      */
      void setPriority(unsigned int);

      bool play();
      bool stop();
      bool pause();
      
      void mute();
      void unmute();

    private:
      enum State { Stopped, Playing, Paused, Invalid };

      Source(ALuint source, ALuint buffer, unsigned int, SourceAllocator*);
      void setupSource();

      ALuint al_src;
      ALuint al_buf;
      State state;
      bool is_muted;
      unsigned int priority;
      bool looping;
      float gain;
      float pitch;
      top10::math::Vector velocity;
      top10::math::Vector position;
      top10::util::Ptr<SourceAllocator> alloc;
    };

    typedef top10::util::Ref< Source > SourceRef;
    typedef std::vector< SourceRef > SourceRefs;

    //! An allocator for openAL sources.
    /*! These sources are pooled and allocated using priorities to cope with
	openAL limits regarding the number of sources.
	*/
    class SourceAllocator: public top10::util::RefCount, private top10::util::NoCopy
    {
    public:
      SourceAllocator(AudioContext*, int num_sources = 16);
      ~SourceAllocator();

      //! Create a new Source, i.e. a candidate owner of openAL sources.
      Source* getNew(ALuint buffer);

      // Used by Source
      //! Try to assign this source to a Resource.
      /*! \return true if success. */
      bool validate(Source*);

    private:
      struct Resource
      {
	ALuint al_source;
	top10::util::Ref<Source> owner;
      };

      top10::util::Ref<AudioContext> audio_context;
      std::vector<Resource> resources;

      void assign(std::vector<Resource>::iterator pos, Source* owner);
    };

    typedef top10::util::Ref< SourceAllocator > SourceAllocatorRef;
    typedef std::vector< SourceAllocatorRef > SourceAllocatorRefs;
  }
}
#endif
