/*
  Top10, a racing simulator
  Copyright (C) 2006  Johann Deneux
  
  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.
  
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
  
  Authors can be contacted at following electronic addresses:
  Johann Deneux: johann.deneux@gmail.com
*/

#include "SourceAllocator.hh"
#include "util/Log.hh"
#include "util/strconv.hh"

#include <cassert>

namespace top10
{
  namespace sound
  {
    Source::Source(ALuint al_src, ALuint al_buf, unsigned int prio, top10::sound::SourceAllocator *alloc)
      : al_src(al_src),
      al_buf(al_buf),
      state(Invalid),
      priority(prio),
      looping(false),
      gain(1.0),
      pitch(1.0),
      alloc(alloc),
      is_muted(false)
    {
    }

    ALuint Source::getSource()
    {
      if (!isValid())
      {
	bool b = alloc->validate(this);
	if (!b)
	  throw InvalidSource();
      }
      return al_src;
    }

    bool Source::isValid() const
    {
      return state != Invalid;
    }

    void Source::setLooping(bool b)
    {
      looping = true;
      if (isValid())
	alSourcei(al_src, AL_LOOPING, b?1:0);
    }

    void Source::setGain(float g)
    {
      gain = g;
      if (isValid())
	alSourcef(al_src, AL_GAIN, g);
    }

    void Source::setPitch(float p)
    {
      pitch = p;
      if (isValid())
	alSourcef(al_src, AL_PITCH, p);
    }

    void Source::setVelocity(const top10::math::Vector& v)
    {
      velocity = v;
      if (isValid())
#if SOUND_VELOCITY_DISABLED
	alSource3f(al_src, AL_VELOCITY, 0.0, 0.0, 0.0);
#else
	alSource3f(al_src, AL_VELOCITY, v.x, v.y, v.z);
#endif
    }

    void Source::setPosition(const top10::math::Vector& p)
    {
      position = p;
      if (isValid())
	alSource3f(al_src, AL_POSITION, p.x, p.y, p.z);
    }

    void Source::setPriority(unsigned int prio)
    {
      priority = prio;
    }

    bool Source::play()
    {
      ALenum al_err;
      try {
	al_err = alGetError();
	ALuint src = getSource();
	ALint status;
	alGetSourcei(src, AL_SOURCE_STATE, &status);
	if (status != AL_PLAYING)
	  if (!is_muted)
	    alSourcePlay(src);

	if (al_err == AL_NO_ERROR)
	{
	  state = Playing;
	  return true;
	}
      }
      catch (InvalidSource)
      {
	return false;
      }
      return false;
    }

    bool Source::pause()
    {
      ALenum al_err;
      try {
	al_err = alGetError();
	ALuint src = getSource();
	alSourcePause(src);
	if (al_err == AL_NO_ERROR)
	{
	  state = Paused;
	  return true;
	}
      }
      catch (InvalidSource)
      {
	return false;
      }
      return false;
    }

    bool Source::stop()
    {
      ALenum al_err;
      try {
	al_err = alGetError();
	ALuint src = getSource();
	alSourceStop(src);
	if (al_err == AL_NO_ERROR)
	{
	  state = Stopped;
	  return true;
	}
      }
      catch (InvalidSource)
      {
	return false;
      }
      return false;
    }

    void Source::setupSource()
    {
      alSourcei(al_src, AL_BUFFER, al_buf);
      alSourcei(al_src, AL_LOOPING, looping?1:0);
      alSourcef(al_src, AL_GAIN, gain);
      alSourcef(al_src, AL_PITCH, pitch);
      alSource3f(al_src, AL_POSITION, position.x, position.y, position.z);
#if SOUND_VELOCITY_DISABLED
      alSource3f(al_src, AL_VELOCITY, 0.0, 0.0, 0.0);
#else
      alSource3f(al_src, AL_VELOCITY, velocity.x, velocity.y, velocity.z);
#endif

    }

    void Source::mute()
    {
      try
      {
	is_muted = true;
	if (state == Playing)
	  alSourceStop(getSource());
      }
      catch (InvalidSource) {}
    }

    void Source::unmute()
    {
      try
      {
	is_muted = false;
	if (state == Playing)
	  alSourcePlay(getSource());
      }
      catch (InvalidSource) {}
    }

    SourceAllocator::SourceAllocator(AudioContext* audio_context, int num_sources)
      : audio_context(audio_context)
    {
      using top10::util::Log;

      ALuint* sources = new ALuint[num_sources];

      bool success = false;
      while(!success && num_sources > 0)
      {
	ALenum al_err;
	alGenSources(num_sources, sources);
	al_err = alGetError();
	if (al_err != AL_NO_ERROR)
	{
	  alDeleteSources(num_sources, sources);
	  num_sources /= 2;
	}
	else
	  success = true;
      }

      if (success)
	Log::getSingle()->send(Log::Info, "SourceAllocator", "Allocated "+top10::util::toString(num_sources)+ " openAL sources");
      else
	Log::getSingle()->send(Log::Warning, "SourceAllocator", "Failed to allocate openAL sources");

      resources.reserve(num_sources);
      for (int i=0; i<num_sources; ++i)
      {
	Resource new_one;
	new_one.al_source = sources[i];
	new_one.owner = 0;
	resources.push_back(new_one);
      }

      delete[] sources;
    }

    SourceAllocator::~SourceAllocator()
    {
      using top10::util::Log;
      Log::getSingle()->send(Log::Info, "SourceAllocator", "Closed");
    }

    Source* SourceAllocator::getNew(ALuint buf)
    {
      return new Source(0, buf, 0, this);
    }

    bool SourceAllocator::validate(Source* owner)
    {
      if (owner->isValid())
	return true;

      // Look for a free spot
      for (std::vector<Resource>::iterator it = resources.begin(); it != resources.end(); ++it)
      {
	// Found one
	if (it->owner.getPtr() == 0)
	{
	  assign(it, owner);
	  return true;
	}
      }

      // Look for a stopped source
      for (std::vector<Resource>::iterator it = resources.begin(); it != resources.end(); ++it)
      {
	assert(it->owner->state != Source::Invalid);

	// Found one
	if (it->owner->state == Source::Stopped)
	{
	  assign(it, owner);
	  return true;
	}
      }

      // No stopped source? Check using openAL
      for (std::vector<Resource>::iterator it = resources.begin(); it != resources.end(); ++it)
      {
	ALint state;
	ALenum al_err = alGetError();

	alGetSourcei(it->al_source, AL_SOURCE_STATE, &state);
	al_err = alGetError();
	if (al_err == AL_NO_ERROR && state == AL_STOPPED)
	{
	  assign(it, owner);
	  return true;
	}
      }

      // Completely full...
      return false;
    }

    void SourceAllocator::assign(std::vector<Resource>::iterator it, top10::sound::Source *owner)
    {
      // Evict previous owner
      if (it->owner.isValid())
	it->owner->state = Source::Invalid;

      // Set the new owner
      owner->al_src = it->al_source;
      owner->state = Source::Stopped;
      it->owner = owner;
      owner->setupSource();
    }
  }
}
