/*
  Top10, a racing simulator
  Copyright (C) 2007  Johann Deneux
  
  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.
  
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
  
  Authors can be contacted at following electronic addresses:
  Johann Deneux: johann.deneux@gmail.com
*/

#include "SkiddingSoundInstance.hh"

namespace top10
{
namespace sound
{
  
  // Dont emit sounds for small speeds (friction ratios are a bit wild at low speeds)
  const float SkiddingSoundInstance::min_speed2 = 0.1;

  SkiddingSoundInstance::SkiddingSoundInstance(SkiddingSound* model, SourceAllocator* alloc)
    : model(model), source_alloc(alloc), enabled(false)
  {
    skidding_source = alloc->getNew(model->getBuffer());
  }

  void SkiddingSoundInstance::setEnabled(bool b)
  {
    if (enabled != b)
    {
      if (b)
	skidding_source->play();
      else
	skidding_source->stop();
    }

    enabled = b;
  }

  void SkiddingSoundInstance::update(float load, float friction, const top10::math::Vector& pos, const top10::math::Vector& speed)
  {
    if (!enabled)
      return;

    if (load < model->getMinLoad() || friction < model->getMinFriction() || speed*speed < min_speed2)
    {
      skidding_source->stop();
      return;
    }

    float gain = 1.0;
    if (load < model->getMaxLoad())
      gain = (load - model->getMinLoad())/(model->getMaxLoad() - model->getMinLoad());

    skidding_source->setGain(gain);
    skidding_source->setPosition(pos);
    skidding_source->setVelocity(speed);
    skidding_source->play();
  }

  void SkiddingSoundInstance::unmute()
  {
    skidding_source->unmute();
  }

  void SkiddingSoundInstance::mute()
  {
    skidding_source->mute();
  }
}
}
