/*
  Top10, a racing simulator
  Copyright (C) 2007  Johann Deneux
  
  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.
  
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
  
  Authors can be contacted at following electronic addresses:
  Johann Deneux: johann.deneux@gmail.com
*/

#include "SkiddingSound.hh"
#include "BufferManager.hh"
#include "util/PathFinder.hh"

namespace top10
{
namespace sound
{

  SkiddingSound::SkiddingSound()
    : top10::util::XmlDumpable("skidding_sound")
  {
    clearState();
  }

  void SkiddingSound::setBuffer(ALuint a)
  {
    al_buf = a;
  }

  ALuint SkiddingSound::getBuffer() const
  {
    return al_buf;
  }

  void SkiddingSound::setMinLoad(float newtons)
  {
    min_load = newtons;
  }
  void SkiddingSound::setMaxLoad(float newtons)
  {
    max_load = newtons;
  }

  float SkiddingSound::getMinLoad() const
  {
    return min_load;
  }

  float SkiddingSound::getMaxLoad() const
  {
    return max_load;
  }

  void SkiddingSound::setMinFriction(float f)
  {
    min_friction = f;
  }

  float SkiddingSound::getMinFriction() const
  {
    return min_friction;
  }

  void SkiddingSound::clearState()
  {
    al_buf = 0;
    min_load = max_load = 0.0;
    min_friction = 0.0;
  }

  int SkiddingSound::saveXml(TiXmlElement *el) const
  {
    std::string buff_name;
    try
    {
      buff_name = BufferManager::getSingle()->lookup(al_buf);
    }
    catch (const BufferManager::BadId&)
    {
      using top10::util::Log;
      Log::getSingle()->send(top10::util::Log::Error, "SkiddingSound/saveXml", "Unknown sound buffer id");
      return -1;
    }

    el->SetAttribute("sound", buff_name.c_str());
    el->SetDoubleAttribute("min", min_load);
    el->SetDoubleAttribute("max", max_load);
    el->SetDoubleAttribute("friction", min_friction);
    return 0;
  }

  int SkiddingSound::loadXml(const TiXmlElement* el)
  {
    const char* buff_name = el->Attribute("sound");
    if (buff_name)
    {
      std::string path = top10::util::PathFinder::defaultPathFinder().find(buff_name);
      if (path.empty())
	path = buff_name;
      al_buf = BufferManager::getSingle()->load(path);
    }
    else
    {
      logXmlNodeError("Missing attribute 'sound'", el, top10::util::Log::Error);
      return -1;
    }

    el->QueryFloatAttribute("min", &min_load);
    el->QueryFloatAttribute("max", &max_load);
    el->QueryFloatAttribute("friction", &min_friction);
    return 0;
  }

  SkiddingSound* buildSkiddingSound()
  {
    using top10::util::Log;

    SkiddingSound* ret = new SkiddingSound;

    // try to load
    std::string path_alt = top10::util::PathFinder::defaultPathFinder().find("kart_data/default/sounds_alt.xml");
    std::string path;
    if (path_alt.empty())
      path = top10::util::PathFinder::defaultPathFinder().find("kart_data/default/sounds.xml");
    else
      path = path_alt;

    if (!path.empty())
    {
      TiXmlDocument xml_doc(path.c_str());
      xml_doc.LoadFile();
      const TiXmlElement *xml_el = xml_doc.RootElement()->FirstChildElement("long_skid");
      if (xml_el)
	xml_el = xml_el->FirstChildElement(ret->getNodeName());

      int s = -1;
      if (xml_el)
	s = ret->load(xml_el);

      if (!s)
	Log::getSingle()->send(Log::Info, "buildSkiddingSound", "Loaded sound specs from file");
      else
	Log::getSingle()->send(Log::Warning, "buildSkiddingSound", "Failed to load sound specs from file");
    }

    return ret;
  }

  SkiddingSound* buildSideSkiddingSound()
  {
    using top10::util::Log;

    SkiddingSound* ret = new SkiddingSound;

    // try to load
    std::string path_alt = top10::util::PathFinder::defaultPathFinder().find("kart_data/default/sounds_alt.xml");
    std::string path;
    if (path_alt.empty())
      path = top10::util::PathFinder::defaultPathFinder().find("kart_data/default/sounds.xml");
    else
      path = path_alt;

    if (!path.empty())
    {
      TiXmlDocument xml_doc(path.c_str());
      xml_doc.LoadFile();
      const TiXmlElement *xml_el = xml_doc.RootElement()->FirstChildElement("side_skid");
      if (xml_el)
	xml_el = xml_el->FirstChildElement(ret->getNodeName());

      int s = -1;
      if (xml_el)
	s = ret->load(xml_el);

      if (!s)
	Log::getSingle()->send(Log::Info, "buildSkiddingSound", "Loaded sound specs from file");
      else
	Log::getSingle()->send(Log::Warning, "buildSkiddingSound", "Failed to load sound specs from file");
    }

    return ret;
  }
}
}
