/*
  Top10, a racing simulator
  Copyright (C) 2007  Johann Deneux
  
  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.
  
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
  
  Authors can be contacted at following electronic addresses:
  Johann Deneux: johann.deneux@gmail.com
*/

#include "RollingSoundInstance.hh"

top10::sound::RollingSoundInstance::RollingSoundInstance(RollingSound* m, SourceAllocator* s)
: model(m), source_alloc(s), enabled(false)
{
  rolling_source = source_alloc->getNew(model->getMainBuffer());
  rolling_source->setLooping(true);

  bool all_done = false;
  int i = 0;
  while (!all_done)
  {
    ALuint buf;
    float freq;

    all_done = !model->getPebble(i, buf, freq);
    if (!all_done)
    {
      SourceRef new_source( source_alloc->getNew(buf) );
      pebble_sources.push_back(new_source);
      pebble_freqs.push_back(freq);
    }
    ++i;
  }
}

void top10::sound::RollingSoundInstance::update(float delta, const top10::math::Vector& pos, const top10::math::Vector& speed_v)
{
  float speed = speed_v.size();

  if (!enabled)
  {
    // Stopping the rolling sound.
    rolling_source->stop();
    // No need to stop the pebbles, their buffers are short and do not loop.
    return;
  }

  // Update the main sound
  float pitch;
  if (speed <= model->getMinSpeed())
    pitch = 0.5;
  else if (speed >= model->getMaxSpeed())
    pitch = 1.0;
  else
    pitch = 0.5 + 0.5*(speed - model->getMinSpeed())/(model->getMaxSpeed() - model->getMinSpeed());
  rolling_source->setPitch(pitch);

  float gain = 1.0;
  if (speed <= model->getMinSpeed())
    gain = speed / model->getMinSpeed();
  rolling_source->setGain(gain);

  rolling_source->setPosition(pos);
  rolling_source->setVelocity(speed_v);
  rolling_source->play();

  // Update the pebble sounds
  assert(pebble_sources.size() == pebble_freqs.size());
  float dist = delta*speed;
  for (int i=0; i<pebble_sources.size(); ++i)
  {
    float f = pebble_freqs.at(i) * dist;
    float r = ((float)rand())/RAND_MAX;
    if (r<=f)
    {
      pebble_sources[i]->setPosition(pos);
      pebble_sources[i]->setVelocity(speed_v);
      pebble_sources[i]->play();
    }
  }
}

void top10::sound::RollingSoundInstance::setEnabled(bool b)
{
  enabled = b;
}

void top10::sound::RollingSoundInstance::unmute()
{
  rolling_source->unmute();
  for (int i=0; i<pebble_sources.size(); ++i)
  {
    pebble_sources[i]->unmute();
  }
}

void top10::sound::RollingSoundInstance::mute()
{
  rolling_source->mute();
  for (int i=0; i<pebble_sources.size(); ++i)
  {
    pebble_sources[i]->mute();
  }
}
