/*
  Top10, a racing simulator
  Copyright (C) 2006  Johann Deneux
  
  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.
  
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
  
  Authors can be contacted at following electronic addresses:
  Johann Deneux: johann.deneux@gmail.com
*/

#ifndef TOP10_SOUND_ROLLING_HH
#define TOP10_SOUND_ROLLING_HH

#include <AL/al.h>
#include <vector>

#include "util/RefCount.hh"
#include "util/NoCopy.hh"

namespace top10
{
  namespace sound
  {
    /*! An instance of this class is a template for sounds emitted by tyres while they are rolling.
        Skidding sounds are not handled here, but driving in the dirt is covered.
	The sound is composed of a main component, emitted continuously. The volume is constant, the
	pitch is constant below min_speed, and above max_speed, interpolated in between.
	Additionally, sounds emitted by pebbles hitting various parts of the vehicle are emitted randomly.
	The frequency of emission of each such sound is controlled by specifying the expected number of hits per meter
	covered by the vehicle. The faster the vehicle moves, the more frequently sounds are triggered.
    */
    class RollingSound: public top10::util::RefCount, private top10::util::NoCopy
    {
    public:
      RollingSound();

      void setMainBuffer(ALuint, float min_speed, float max_speed);
      ALuint getMainBuffer() const;
      float getMinSpeed() const;
      float getMaxSpeed() const;

      void addPebbleBuffer(ALuint, float hits_per_meter);
      //! Return the buffer and average hits per meter for the idx-th entry.
      /*! \return false if idx is out of range. */
      bool getPebble(int idx, ALuint&, float&) const;
      
    private:
      float min_speed;
      float max_speed;
      ALuint rolling_buf;

      struct PebbleHit
      {
	ALuint buf;
	//! Average number of hits per meter
	float hits;
      };
      std::vector<PebbleHit> pebbles;
    };

    typedef top10::util::Ref< RollingSound > RollingSoundRef;
    typedef std::vector< RollingSoundRef > RollingSoundRefs;

    RollingSound* makeRollingSound();

  }
}
#endif
