/*
  Top10, a racing simulator
  Copyright (C) 2006  Johann Deneux
  
  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.
  
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
  
  Authors can be contacted at following electronic addresses:
  Johann Deneux: johann.deneux@gmail.com
*/

#include "EngineSoundInstance.hh"
#include "BufferManager.hh"
#include "AudioContext.hh"

#include "util/Log.hh"

using top10::util::Log;

namespace top10
{
  namespace sound
  {
    EngineSoundInstance::EngineSoundInstance(const top10::sound::EngineSound* model, SourceAllocator* alloc)
      : model(model), source_alloc(alloc), error_reported(false)
    {
      std::vector<ALuint> buffers;
      std::vector<ALuint>::const_iterator it;

      buffers = model->getBuffersOn();

      ALenum al_err;
      al_err = alGetError();
      if (al_err != AL_NO_ERROR)
	Log::getSingle()->send(Log::Warning, "EngineSoundInstance",
	std::string("AL error ") + top10::sound::makeErrorStr(al_err) + " before creating sources ");

      for (it = buffers.begin(); it != buffers.end(); ++it)
      {
	SourceRef new_source( source_alloc->getNew(*it) );
	new_source->setLooping(true);
	al_sources_on.push_back(new_source);
      }

      buffers = model->getBuffersOff();
      for (it = buffers.begin(); it != buffers.end(); ++it)
      {
	SourceRef new_source( source_alloc->getNew(*it) );
	new_source->setLooping(true);
	al_sources_off.push_back(new_source);
      }

      al_err = alGetError();
      if (al_err != AL_NO_ERROR)
	Log::getSingle()->send(Log::Warning, "EngineSoundInstance",
	std::string("AL error ") + top10::sound::makeErrorStr(al_err) + " after creating sources ");

    }

    void EngineSoundInstance::update(double rpm, bool on_load, const top10::math::Vector& pos, const top10::math::Vector& speed)
    {
      SourceRefs* active;
      SourceRefs* inactive;
    
      if (on_load)
      {
	active = &al_sources_on;
	inactive = &al_sources_off;
      }
      else
      {
	active = &al_sources_off;
	inactive = &al_sources_on;
      }

      ALenum al_err;
      al_err = alGetError();
      if (!error_reported && al_err != AL_NO_ERROR)
      {
	error_reported = true;
	Log::getSingle()->send(Log::Warning, "EngineSoundInstance/update",
	std::string("AL error ") + top10::sound::makeErrorStr(al_err) + " before alSourceStop (inactive).");
      }

      for (int i = 0; i != inactive->size(); ++i)
      {
	(*inactive)[i]->stop();
      }

      std::vector<double> gains;
      std::vector<double> pitches;
      model->getSourceParameters(rpm, on_load, gains, pitches);

      std::vector<double>::size_type i = 0;
      for (int it = 0; it != active->size(); ++it, ++i)
      {
	try
	{
	  if (gains.at(i) > 0.0 && pitches.at(i) > 0.0)
	  {
	    al_err = alGetError();
	    if (!error_reported && al_err != AL_NO_ERROR)
	    {
	      error_reported = true;
	      Log::getSingle()->send(Log::Warning, "EngineSoundInstance/update",
		std::string("AL error ") + top10::sound::makeErrorStr(al_err) + " before alSource (GAIN/PITCH).");
	    }

	    // Gain and pitch
	    (*active)[it]->setGain(gains.at(i));
	    (*active)[it]->setPitch(pitches.at(i));

	    // Position and velocity
	    (*active)[it]->setPosition(pos);
	    (*active)[it]->setVelocity(speed);

	    al_err = alGetError();
	    if (!error_reported && al_err != AL_NO_ERROR)
	    {
	      error_reported = true;
	      Log::getSingle()->send(Log::Warning, "EngineSoundInstance/update",
		std::string("AL error ") + top10::sound::makeErrorStr(al_err) + " before alGetSource(AL_SOURCE_STATE).");
	    }

	    (*active)[it]->play();
	  }
	  else
	  {
	    (*active)[it]->stop();
	  }
	}
	catch (Source::InvalidSource)
	{
	}
      }
    }

    void EngineSoundInstance::unmute()
    {
      for (int i=0; i < al_sources_on.size(); ++i)
	al_sources_on[i]->unmute();

      for (int i=0; i < al_sources_off.size(); ++i)
	al_sources_off[i]->unmute();
    }

    void EngineSoundInstance::mute()
    {
      for (int i=0; i < al_sources_on.size(); ++i)
	al_sources_on[i]->mute();

      for (int i=0; i < al_sources_off.size(); ++i)
	al_sources_off[i]->mute();
    }

  };
};
