/*
  Top10, a racing simulator
  Copyright (C) 2006  Johann Deneux
  
  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.
  
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
  
  Authors can be contacted at following electronic addresses:
  Johann Deneux: johann.deneux@gmail.com
*/

#include "EngineSound.hh"
#include "BufferManager.hh"
#include "util/PathFinder.hh"
#include "util/Log.hh"

using top10::util::Log;

namespace top10
{
  namespace sound
  {
    EngineRpmRange::EngineRpmRange()
      : top10::util::XmlDumpable("range")
    {
      clearState();
    }

    EngineRpmRange::EngineRpmRange(ALuint buffer, double rpm_min, double rpm_start, double rpm_end, double rpm_max,
      double pitch_min, double pitch_max)
      : top10::util::XmlDumpable("range"),
      al_buffer(buffer),
      rpm_min(rpm_min),
      rpm_start(rpm_start),
      rpm_end(rpm_end),
      rpm_max(rpm_max),
      pitch_min(pitch_min),
      pitch_max(pitch_max)
    {
    }

    ALuint EngineRpmRange::getBuffer() const
    {
      return al_buffer;
    }

    double EngineRpmRange::getRpmEnd() const
    {
      return rpm_end;
    }

    double EngineRpmRange::getRpmStart() const
    {
      return rpm_start;
    }

    double EngineRpmRange::getRpmMin() const
    {
      return rpm_min;
    }

    double EngineRpmRange::getRpmMax() const
    {
      return rpm_max;
    }

    double EngineRpmRange::getPitchMin() const
    {
      return pitch_min;
    }

    double EngineRpmRange::getPitchMax() const
    {
      return pitch_max;
    }

    double EngineRpmRange::getPitch(double rpm) const
    {
      double ret = 1.0;

      if (rpm <= rpm_min)
	ret = pitch_min;
      else if (rpm >= rpm_max)
	ret = pitch_max;
      else
	ret = (pitch_max - pitch_min) * (rpm - rpm_min) / (rpm_max - rpm_min) + pitch_min;

      return ret;
    }

    void EngineRpmRange::clearState()
    {
      al_buffer = 0;
      rpm_min = rpm_start = rpm_end = rpm_max = 0.0;
      pitch_min = 1.0;
      pitch_max = 1.0;
    }

    int EngineRpmRange::saveXml(TiXmlElement *el) const
    {
      std::string buff_name;
      try
      {
	buff_name = BufferManager::getSingle()->lookup(al_buffer);
      }
      catch (const BufferManager::BadId&)
      {
	using top10::util::Log;
	Log::getSingle()->send(top10::util::Log::Error, "engineRpmRange/saveXml", "Unknown sound buffer id");
	return -1;
      }

      el->SetAttribute("sound", buff_name.c_str());
      el->SetDoubleAttribute("min", rpm_min);
      el->SetDoubleAttribute("start", rpm_start);
      el->SetDoubleAttribute("end", rpm_end);
      el->SetDoubleAttribute("max", rpm_max);
      el->SetDoubleAttribute("pitch_min", pitch_min);
      el->SetDoubleAttribute("pitch_max", pitch_max);

      return 0;
    }

    int EngineRpmRange::loadXml(const TiXmlElement* el)
    {
      const char* buff_name = el->Attribute("sound");
      if (buff_name)
      {
	std::string path = top10::util::PathFinder::defaultPathFinder().find(buff_name);
	if (path.empty())
	  path = buff_name;
	al_buffer = BufferManager::getSingle()->load(path);
      }
      else
      {
	logXmlNodeError("Missing attribute 'sound'", el, top10::util::Log::Error);
	return -1;
      }

      el->QueryDoubleAttribute("min", &rpm_min);
      el->QueryDoubleAttribute("start", &rpm_start);
      el->QueryDoubleAttribute("end", &rpm_end);
      el->QueryDoubleAttribute("max", &rpm_max);
      el->QueryDoubleAttribute("pitch_min", &pitch_min);
      el->QueryDoubleAttribute("pitch_max", &pitch_max);

      return 0;
    }

    EngineSound::EngineSound()
      : top10::util::XmlDumpable("engine_sound")
    {
    }

    void EngineSound::insertRangeOn(top10::sound::EngineRpmRange r)
    {
      std::vector<EngineRpmRange>::iterator it;
      for (it = ranges_on.begin(); it != ranges_on.end() && it->getRpmStart() < r.getRpmStart(); ++it);
      ranges_on.insert(it, r);
    }

    void EngineSound::insertRangeOff(top10::sound::EngineRpmRange r)
    {
      std::vector<EngineRpmRange>::iterator it;
      for (it = ranges_off.begin(); it != ranges_off.end() && it->getRpmStart() < r.getRpmStart(); ++it);
      ranges_off.insert(it, r);
    }

    std::vector<ALuint> EngineSound::getBuffersOn() const
    {
      std::vector<ALuint> ret;
      for (std::vector<EngineRpmRange>::const_iterator it = ranges_on.begin(); it != ranges_on.end(); ++it)
	ret.push_back(it->getBuffer());
      return ret;
    }

    std::vector<ALuint> EngineSound::getBuffersOff() const
    {
      std::vector<ALuint> ret;
      for (std::vector<EngineRpmRange>::const_iterator it = ranges_off.begin(); it != ranges_off.end(); ++it)
	ret.push_back(it->getBuffer());
      return ret;
    }

    void EngineSound::getSourceParameters(double rpm, bool on_load, std::vector<double> &amplitudes, std::vector<double> &pitches) const
    {
      const std::vector<EngineRpmRange>* ranges = 0;
      if (on_load) ranges = &ranges_on;
      else         ranges = &ranges_off;

      for (std::vector<EngineRpmRange>::const_iterator it = ranges->begin(); it != ranges->end(); ++it)
      {
	// For all ranges finishing before rpm, disable the source.
	if (it->getRpmMax() < rpm)
	{
	  amplitudes.push_back(0.0);
	  pitches.push_back(0.0);
	}

	// For all ranges fading, compute the attenuation of the amplitude.
	else if (it->getRpmEnd() < rpm)
	{
	  double f = rpm - it->getRpmEnd();
	  f /= it->getRpmMax() - it->getRpmEnd();
	  f = 1.0 - f;
	  if (f < 0.0)
	    f = 0.0;
	  if (f > 1.0)
	    f = 1.0;
	  amplitudes.push_back(f);

	  pitches.push_back(it->getPitch(rpm));
	}

	// For ranges in full, set full amplitude
	else if (it->getRpmStart() < rpm)
	{
	  amplitudes.push_back(1.0);

	  pitches.push_back(it->getPitch(rpm));
	}

	// For rising ranges
	else if (it->getRpmMin() < rpm)
	{
	  double f = rpm - it->getRpmMin();
	  f /= it->getRpmStart() - it->getRpmMin();
	  if (f < 0.0)
	    f = 0.0;
	  if (f > 1.0)
	    f = 1.0;
	  amplitudes.push_back(f);

	  pitches.push_back(it->getPitch(rpm));
	}

	// Ranges not enabled yet
	else
	{
	  amplitudes.push_back(0.0);
	  pitches.push_back(0.0);
	}
      }
    }

    void EngineSound::clearState()
    {
      ranges_on.clear();
      ranges_off.clear();
    }

    int EngineSound::loadXml(const TiXmlElement* el)
    {
      EngineRpmRange tmp;
      const TiXmlElement* child;

      child = el->FirstChildElement("loaded");
      if (child)
	child = child->FirstChildElement(tmp.getNodeName());
      while (child)
      {
	tmp.load(child);
	insertRangeOn(tmp);
	child = child->NextSiblingElement(tmp.getNodeName());
      }

      child = el->FirstChildElement("unloaded");
      if (child)
	child = child->FirstChildElement(tmp.getNodeName());
      while (child)
      {
	tmp.load(child);
	insertRangeOff(tmp);
	child = child->NextSiblingElement(tmp.getNodeName());
      }

      return 0;
    }

    int EngineSound::saveXml(TiXmlElement* el) const
    {
      TiXmlElement loaded("loaded");
      for (std::vector<EngineRpmRange>::const_iterator it = ranges_on.begin(); it != ranges_on.end(); ++it)
      {
	TiXmlElement tmp("tmp");
	it->save(&tmp);
	loaded.InsertEndChild(tmp);
      }

      TiXmlElement unloaded("unloaded");
      for (std::vector<EngineRpmRange>::const_iterator it = ranges_off.begin(); it != ranges_off.end(); ++it)
      {
	TiXmlElement tmp("tmp");
	it->save(&tmp);
	unloaded.InsertEndChild(tmp);
      }

      el->InsertEndChild(loaded);
      el->InsertEndChild(unloaded);

      return 0;
    }

    EngineRpmRange EngineSound::loadData(const char *filename, double rpm_min, double rpm_start, double rpm_end, double rpm_max,
      double pitch_min, double pitch_max)
    {
      ALuint al_buffer = 0;

      std::string path = top10::util::PathFinder::defaultPathFinder().find(filename);
      if (path.empty())
	throw top10::util::Error(std::string("Could not find ") + std::string(filename));

      al_buffer = BufferManager::getSingle()->load(path);
      return EngineRpmRange(al_buffer, rpm_min, rpm_start, rpm_end, rpm_max, pitch_min, pitch_max);
    }

    EngineSound* buildKartSound()
    {
      EngineSound* ret = new EngineSound;

      // try to load
      std::string path_alt = top10::util::PathFinder::defaultPathFinder().find("kart_data/default/sounds_alt.xml");
      std::string path;
      if (path_alt.empty())
	path = top10::util::PathFinder::defaultPathFinder().find("kart_data/default/sounds.xml");
      else
	path = path_alt;

      if (!path.empty())
      {
	TiXmlDocument xml_doc(path.c_str());
	xml_doc.LoadFile();
	const TiXmlElement *xml_el = xml_doc.RootElement()->FirstChildElement(ret->getNodeName());

	int s = -1;
	if (xml_el)
	  s = ret->load(xml_el);

	if (!s)
	  Log::getSingle()->send(Log::Info, "buildKartSound", "Loaded sound specs from file");
	else
	  Log::getSingle()->send(Log::Warning, "buildKartSound", "Failed to load sound specs from file");
      }
      // create default sound otherwise
      else
      {
	try {
	  EngineRpmRange rng = EngineSound::loadData("idle.wav", 0.0, 1.0, 5.0, 10.0, 0.75, 1.0);
	  ret->insertRangeOff(rng);
	}
	catch(const top10::util::Error& err)
	{
	  Log::getSingle()->send(Log::Warning, "buildKartSound", std::string("loading idle.wav: ") + err);
	}

	try {
	  EngineRpmRange rng = EngineSound::loadData("low_off.wav", 5.0, 10.0, 50.0, 60.0, 0.75, 1.0);
	  ret->insertRangeOff(rng);
	}
	catch(const top10::util::Error& err)
	{
	  Log::getSingle()->send(Log::Warning, "buildKartSound", std::string("loading low_off.wav: ") + err);
	}

	try {
	  EngineRpmRange rng = EngineSound::loadData("mid_off.wav", 50.0, 60.0, 100.0, 110.0, 0.75, 1.0);
	  ret->insertRangeOff(rng);
	}
	catch(const top10::util::Error& err)
	{
	  Log::getSingle()->send(Log::Warning, "buildKartSound", std::string("loading mid_off.wav: ") + err);
	}

	try {
	  EngineRpmRange rng = EngineSound::loadData("hi_off.wav", 100.0, 110.0, 200.0, 210.0, 0.75, 1.0);
	  ret->insertRangeOff(rng);
	}
	catch(const top10::util::Error& err)
	{
	  Log::getSingle()->send(Log::Warning, "buildKartSound", std::string("loading hi_off.wav: ") + err);
	}


	try {
	  EngineRpmRange rng = EngineSound::loadData("idle.wav", 0.0, 1.0, 5.0, 10.0, 0.75, 1.0);
	  ret->insertRangeOn(rng);
	}
	catch(const top10::util::Error& err)
	{
	  Log::getSingle()->send(Log::Warning, "buildKartSound", std::string("loading idle.wav: ") + err);
	}

	try {
	  EngineRpmRange rng = EngineSound::loadData("low_on.wav", 5.0, 10.0, 50.0, 60.0, 0.75, 1.0);
	  ret->insertRangeOn(rng);
	}
	catch(const top10::util::Error& err)
	{
	  Log::getSingle()->send(Log::Warning, "buildKartSound", std::string("loading low_on.wav: ") + err);
	}

	try {
	  EngineRpmRange rng = EngineSound::loadData("mid_on.wav", 50.0, 60.0, 100.0, 110.0, 0.75, 1.0);
	  ret->insertRangeOn(rng);
	}
	catch(const top10::util::Error& err)
	{
	  Log::getSingle()->send(Log::Warning, "buildKartSound", std::string("loading mid_on.wav: ") + err);
	}

	try {
	  EngineRpmRange rng = EngineSound::loadData("hi_on.wav", 100.0, 110.0, 200.0, 210.0, 0.75, 1.0);
	  ret->insertRangeOn(rng);
	}
	catch(const top10::util::Error& err)
	{
	  Log::getSingle()->send(Log::Warning, "buildKartSound", std::string("loading hi_on.wav: ") + err);
	}
      }

      return ret;
    }
  };
};
