/*
  Top10, a racing simulator
  Copyright (C) 2006  Johann Deneux
  
  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.
  
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
  
  Authors can be contacted at following electronic addresses:
  Johann Deneux: johann.deneux@gmail.com
*/
#include <SDL/SDL.h>   // To load samples
#include <map>

#include "BufferManager.hh"
#include "AudioContext.hh"
#include "util/Log.hh"
#include "util/error.hh"

using top10::util::Log;

namespace top10
{
  namespace sound
  {
    top10::util::Ptr<BufferManager> BufferManager::single;

    BufferManager* BufferManager::getSingle()
    {
      if (single.isValid())
	return single.getPtr();

      single = new BufferManager;
      single->use(); //quick fix: to avoid that the last Ref<BufferManager> deallocate single.

      return single.getPtr();
    }

    BufferManager::~BufferManager()
    {
      for (std::map<std::string, ALuint>::iterator it = buffers.begin(); it != buffers.end(); ++it)
	alDeleteBuffers(1, &(it->second));
    }

    ALuint BufferManager::load(const std::string path)
    {
      std::map<std::string, ALuint>::const_iterator find_it = buffers.find(path);
      if (find_it != buffers.end())
	return find_it->second;

      /* sound samples specs */
      SDL_AudioSpec audiospec;
      Uint32 len;
      Uint8* buff;
      ALuint al_buffer = 0;

      if (!SDL_LoadWAV(path.c_str(), &audiospec, &buff, &len))
	throw top10::util::Error(SDL_GetError());

      ALenum al_err = alGetError();
      if (al_err != AL_NO_ERROR)
	Log::getSingle()->send(Log::Warning, "BufferManager/load",
	std::string("AL error ") + top10::sound::makeErrorStr(al_err) + " before loading " + path);

      alGenBuffers(1, &al_buffer);

      /*
       * Convert to an openAL format.
       * OpenAL is vague about its supported format. Endianness? Signed vs unsigned?
       */
      ALenum al_format;
      bool conv_ok = audiospec.channels <= 2;
      switch(audiospec.format)
      {
      case AUDIO_U8:
      case AUDIO_S8:
	  al_format = audiospec.channels == 1? AL_FORMAT_MONO8 : AL_FORMAT_STEREO8;
	  break;
      case AUDIO_U16:
      case AUDIO_S16:
	  al_format = audiospec.channels == 1? AL_FORMAT_MONO16 : AL_FORMAT_STEREO16;
	  break;
      default:
	  conv_ok = false;
	  break;
      }
      if (conv_ok)
	  alBufferData(al_buffer, al_format, (void*)buff, len, audiospec.freq);
      else
	  Log::getSingle()->send(Log::Error, "BufferManager/load", path + " uses an unsupported format");

      al_err = alGetError();
      if (al_err != AL_NO_ERROR)
	Log::getSingle()->send(Log::Warning, "BufferManager/load",
	std::string("AL error ") + top10::sound::makeErrorStr(al_err) + " after loading " + path);

      SDL_FreeWAV(buff);

      buffers[path] = al_buffer;
      return al_buffer;
    }

    std::string BufferManager::lookup(ALuint id) const
    {
      for (std::map<std::string, ALuint>::const_iterator it = buffers.begin();
	it != buffers.end();
	++it)
      {
	if (it->second == id)
	  return it->first;
      }
      throw BadId();
      return "";
    }
  }
}
