/*
  Top10, a racing simulator
  Copyright (C) 2006  Johann Deneux
  
  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.
  
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
  
  Authors can be contacted at following electronic addresses:
  Johann Deneux: johann.deneux@gmail.com
*/

#include "AudioContext.hh"
#include "util/Log.hh"

using top10::util::Log;

namespace top10
{
namespace sound
{
  AudioContext::Error::Error(const std::string& msg)
    : top10::util::Error(msg)
  {
  }

  AudioContext::AudioContext(const char* device_name)
    : dev(0), context(0)
  {
    std::string dev_name;
    if (device_name)
      dev_name = device_name;
    else
      dev_name = "(default)";

    ALenum al_err;
    al_err = alGetError();
    if (al_err != AL_NO_ERROR)
      Log::getSingle()->send(Log::Warning, "AudioContext/getSingle",
      "AL error " + makeErrorStr(al_err) + " before alcOpenDevice.");

    dev = alcOpenDevice(device_name);

    al_err = alGetError();
    if (al_err != AL_NO_ERROR)
      throw Error("AL error " + makeErrorStr(al_err) + " when opening output sound device "+dev_name);

    if (dev)
    {      
      context = alcCreateContext(dev, 0);

      al_err = alGetError();
      if (al_err != AL_NO_ERROR)
      {
	alcCloseDevice(dev);
	throw Error("AL error " + makeErrorStr(al_err) + " when creating context");
      }

      if (context)
	alcMakeContextCurrent(context);

      al_err = alGetError();
      if (al_err != AL_NO_ERROR)
	Log::getSingle()->send(Log::Warning, "AudioContext/getSingle",
	"AL error " + makeErrorStr(al_err) + " after alcMakeContextCurrent.");

    }
  }

  AudioContext::~AudioContext()
  {
    alcDestroyContext(context);
    alcCloseDevice(dev);
  }


  std::string makeErrorStr(ALenum err)
  {
    switch(err)
    {
    case AL_NO_ERROR:          return std::string("AL_NO_ERROR");          break;
    case AL_INVALID_NAME:      return std::string("AL_INVALID_NAME");      break;
    case AL_INVALID_ENUM:      return std::string("AL_INVALID_ENUM");      break;
    case AL_INVALID_VALUE:     return std::string("AL_INVALID_VALUE");     break;
    case AL_INVALID_OPERATION: return std::string("AL_INVALID_OPERATION"); break;
    case AL_OUT_OF_MEMORY:     return std::string("AL_OUT_OF_MEMORY");     break;
    default:                   return std::string("AL_???");               break;
    }
    return std::string("<UNREACHABLE>");
  }

};
};
