/*
  Top10, a racing simulator
  Copyright (C) 2000-2004  Johann Deneux
  
  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.
  
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
  
  Authors can be contacted at following electronic addresses:
  Johann Deneux: johann.deneux@gmail.com
*/

#include "Simulation.hh"
#include "util/SlowTime.hh"
#include "util/GlobalOptions.hh"
#include <algorithm>

using namespace top10::math;
using namespace top10::racing;
using namespace std;

Simulation::Simulation(top10::track::Track* _track,
		       const top10::physX::KartRefs& _karts,
		       LapRecordDb* laps):
  track(_track),
  karts(),
  isWet(false),
  current_lap_records(_karts.size()),
  saved_laps(laps),
  steer_angles(_karts.size()),
  throttles(_karts.size()),
  brakings(_karts.size()),
  gears(_karts.size()),
  current_t(0)
{
  world.fixed_objects.push_back(track.getPtr());
  
  // Put the karts on the track
  for (int i=0; i<_karts.size(); ++i) {
    Vector pos;
    Vector dir;
    track->getStartingArea()->getStartingPos(i, pos, dir);
    top10::physX::Kart* kart = _karts[i].getPtr();
    kart->setPosOnTrack(pos, dir);
    setKart(i, kart);
  }
  
  lap_record_period = (int)(1000.0*getOptD("Record.Period"));
  last_record_update = 0;
}

void Simulation::setKart(unsigned int id, top10::physX::Kart* kart)
{
  top10::physX::Kart* old_kart = 0;
  if (id < karts.size())
    old_kart = karts[id].getPtr();

  std::vector<top10::physX::ComplexObject*>::iterator find_it = std::find(world.objects.begin(), world.objects.end(), old_kart);
  if (find_it != world.objects.end())
    *find_it = kart;
  else
    world.objects.push_back(kart);

  assert( karts.size() >= id );
  top10::physX::KartRef kref( kart );
  if (id < karts.size())
    karts.at(id) = kref;
  else
    karts.push_back(kref);

  assert( timers.size() >= id );
  if (id < timers.size())
  {
    timers[id] = KartTimer(kart, track->getCheckPoints());
    steer_angles.at(id) = 0;
    throttles.at(id) = 0;
    brakings.at(id) = 0;
    gears.at(id) = 0;
  }
  else
  {
    timers.push_back(KartTimer(kart, track->getCheckPoints()));
    steer_angles.push_back(0);
    throttles.push_back(0);
    brakings.push_back(0);
    gears.push_back(0);
  }
}


void Simulation::sendEvent(int kart_i, Action action, Sint16 value)
{
  switch (action) {
  case Steering:
    steer_angles.at(kart_i) = -value/32768.0;
    break;
  case Acceleration:
    throttles.at(kart_i) = (value+32768)/65536.0;
    break;
  case Braking:
    brakings.at(kart_i) = (value+32768)/65536.0;
    break;
  case AccelerationAndBraking:
    if (value > 0) {
      throttles.at(kart_i) = value/32768.0;
      brakings.at(kart_i) = 0.0;
    }
    else {
      throttles.at(kart_i) = 0.0;
      brakings.at(kart_i) = -value/32768.0;
    }
    break;
  case BrakingAndAcceleration:
    if (value < 0) {
      throttles.at(kart_i) = -value/32768.0;
      brakings.at(kart_i) = 0.0;
    }
    else {
      throttles.at(kart_i) = 0.0;
      brakings.at(kart_i) = value/32768.0;
    }
    break;
  case GearDown:
    if (value < 0) {  // Change gear when the gear shifter is released
      gears.at(kart_i)--;
      karts[kart_i]->setGear(gears.at(kart_i));
      gears.at(kart_i) = karts[kart_i]->getGear();
    }
    
    break;
  case GearUp:
    if (value < 0) {  // Change gear when the gear shifter is released
      gears.at(kart_i)++;
      karts[kart_i]->setGear(gears.at(kart_i));
      gears.at(kart_i) = karts[kart_i]->getGear();
    }
    break;

  case ResetKart:
  {
      top10::physX::Kart* kart = karts[kart_i].getPtr();
      top10::physX::Kart::State state = kart->getKartState();
      if (state.speed.size() < 0.1)
      {
	  top10::math::Vector pos = kart->getPos();
	  top10::math::Vector dir = state.orient * top10::math::Vector(1.0, 0.0, 0.0);
	  kart->setPosOnTrack(pos, dir);
      }
  }
      break;

  default: abort();  /* We should not receive other kinds of events */
  }
}

void Simulation::update(unsigned int now)
{
  assert(now >= current_t);
  assert(now >= last_record_update);
  
  for (int i = 0; i < karts.size(); ++i)
  {
      karts[i]->steer(steer_angles.at(i));
      karts[i]->accel(throttles.at(i));
      karts[i]->brake(brakings.at(i));
  }

  while (current_t + top10::physX::World::time_slice < now) {
    // Update physics
    world.simulate();
    current_t += top10::physX::World::time_slice;

    // Record the kart states, update the kart timers.
    std::vector<LapRecord>::iterator lap_it = current_lap_records.begin();
    std::vector<top10::racing::KartTimer>::iterator timer_it = timers.begin();
    int kart_i = 0;
    for( ;lap_it != current_lap_records.end(); ++lap_it, ++kart_i, ++timer_it) {
      assert(kart_i <= karts.size());
      assert(timer_it != timers.end());

      timer_it->update(top10::physX::World::time_slice);

      if (timer_it->timerRunning() && (current_t-last_record_update > lap_record_period || timer_it->justFinishedLap())) {
	KartState s = karts.at(kart_i)->getKartState();
	s.timestamp = current_t;

	lap_it->addState(s);
	last_record_update = current_t;

	if (timer_it->justFinishedLap()) {
	  std::cerr<<"Just finished lap"<<std::endl;
	  lap_it->setDate( top10::util::Date::now() );
	  saved_laps->addLap(*lap_it, track->getName(), karts[kart_i]->getName(), isWet);
	  if (best_lap_record.size() == 0 || lap_it->getTime() < best_lap_record.getTime()) {
	    std::cerr<<"New best lap "<<lap_it->getTime()<<" < "<<best_lap_record.getTime()<<std::endl;
	    best_lap_record.swap(*lap_it);
	  }
	  else {
	    std::cerr<<"Sucky lap"<<lap_it->getTime()<<" >= "<<best_lap_record.getTime()<<std::endl;
	  }
	  lap_it->clear(current_t);
	}
      }
    }
  }
}

