/*
  Top10, a racing simulator
  Copyright (C) 2000-2007  Johann Deneux
  
  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.
  
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
  
  Authors can be contacted at following electronic addresses:
  Johann Deneux: johann.deneux@gmail.com
*/

#ifndef TOP10_RACING_LAP_RECORD_DB_HH
#define TOP10_RACING_LAP_RECORD_DB_HH

#include "util/NoCopy.hh"
#include "LapRecord.hh"

namespace top10
{
  namespace racing
  {
    //! An entry in the database record.
    class LapRecordEntry
    {
    public:
      LapRecordEntry(const top10::racing::LapRecord lap,
		     const std::string& track_name,
		     const std::string& kart_name,
		     bool is_foggy,
		     bool is_saved);

      //! Return a human-readable partial representation.
      std::string getRepr() const;

      top10::racing::LapRecord getLap() const;
      std::string getTrackName() const;
      std::string getKartName() const;
      bool isFoggy() const;

    private:
      //! The recorded lap.
      top10::racing::LapRecord m_lap;
      //! Name of the track.
      std::string m_track_name;
      //! Name of the vehicle.
      std::string m_kart_name;
      //! Wet weather.
      bool m_is_foggy;
      //! True if the lap exists on disk.
      bool m_is_saved;
    };



    //! A database of laps.
    class LapRecordDb: public top10::util::NoCopy
    {
    public:
      //! Load a lap.
      void addSavedLap(const std::string& filename);
      //! Add a lap from memory.
      void addLap(const top10::racing::LapRecord&,
		  const std::string& track_name,
		  const std::string& kart_name,
		  bool foggy);

      unsigned int getNumEntries() const;
      LapRecordEntry getEntry(unsigned int) const;

      //! Write a lap to disk.
      void saveLap(unsigned int);

      //! Remove a lap from the database.
      /*! Non-saved laps are lost. */
      void discardLap(unsigned int);

      //! Remove a lap and delete from disk if it was saved.
      void deleteLap(unsigned int);

    private:
      std::vector<LapRecordEntry> m_entries;
    };

  }
}

#endif
