/*
  Top10, a racing simulator
  Copyright (C) 2000-2004  Johann Deneux
  
  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.
  
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
  
  Authors can be contacted at following electronic addresses:
  Johann Deneux: johann.deneux@it.uu.se
*/

#ifndef TOP10_LAPRECORD_HH
#define TOP10_LAPRECORD_HH

#include "math/Matrix.hh"
#include "physX/Kart.hh"
#include "util/Date.hh"

#include <vector>
#include <cassert>

namespace top10 {
  namespace racing {
    typedef top10::physX::Kart::State KartState;

    //! Record a full lap
    class LapRecord : top10::util::Dumpable
    {
    public:
      typedef std::vector<KartState> ContainerT;
      typedef ContainerT::const_iterator const_iterator;

      LapRecord(): lap_start(0), hint(0) {}

      inline void setDate(const top10::util::Date& d) { date = d; }
      inline top10::util::Date getDate() const { return date; }

      /*!
	\note s.timestamp is absolute, but the stored value is relative to the
	start of this lap.
      */
      inline void addState(KartState s) {
	assert(s.timestamp >= lap_start);
        if (lap_start == 0) lap_start = s.timestamp;
	s.timestamp -= lap_start;
	states.push_back(s);
      }
      inline const_iterator begin() const { return states.begin(); }
      inline const_iterator end() const { return states.end(); }

      //! return the state of the kart at time ts.
      /*! \param ts milliseconds since start of lap */
      KartState getKartState(unsigned int ts) const;
      
      inline unsigned int getTime() const {
	if (states.empty()) return 0;
	else return states.rbegin()->timestamp;
      }

      inline unsigned int getStartTime() const { return lap_start; }

      inline void swap(LapRecord& other) {
	std::swap(date, other.date);
	states.swap(other.states);
	std::swap(lap_start, other.lap_start);
        hint = 0;
      }

      inline void clear(unsigned int start) {
	states.clear();
	lap_start = start;
	date = top10::util::Date();
        hint = 0;
      }

      inline int size() const {
	return states.size();
      }

      void dumpTo(std::ostream& out) const;
      void loadFrom(std::istream& in);

    private:
      ContainerT states;
      //! Timestamp of the start of the lap
      unsigned int lap_start;
      //! Date of the lap.
      top10::util::Date date;
      
      //! used by getKartState to quickly find the states closest to a given timestamp
      mutable ContainerT::size_type hint;
    };

  };
};

#endif
