/*
  Top 10, a racing simulator
  Copyright (C) 2003,2005  Johann Deneux
  
  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.
  
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
  
  Authors can be contacted at following electronic addresses:
  Johann Deneux: johann.deneux@it.uu.se
*/

#include "LapRecord.hh"

namespace top10 {
namespace racing {
  
KartState LapRecord::getKartState(unsigned int ts) const
{
  KartState ret;
  ret.orient = top10::math::Identity3();
  
  if (states.empty()) return ret;
  
  // Move hint to some position earlier than ts
  while (states.at(hint).timestamp > ts && hint != 0) --hint;
  
  // "after" will point on the state with smallest timestamp above ts
  ContainerT::size_type after = hint;
  while (after != states.size() && states.at(after).timestamp < ts) {
    ++after;
  }
  if (after == states.size()) --after;
  hint = after;
  
  ContainerT::size_type before = after;
  if (before != 0) --before;
  
  unsigned int dt = states.at(after).timestamp - states.at(before).timestamp;
  float x = (float)ts-states.at(before).timestamp;
  if (dt != 0) x /= dt;
  else x = 0.0;
  float y = 1.0-x;
  
  ret.steer = y*states.at(before).steer + x*states.at(after).steer;
  ret.orient = y*states.at(before).orient + x*states.at(after).orient;   // Linear interpolation of rotation matrices is not correct, but it looks ok.
  ret.timestamp = ts;
  ret.translation = y*states.at(before).translation + x*states.at(after).translation;
  ret.speed = y*states.at(before).speed + x*states.at(after).speed;
  ret.throttle = y*states.at(before).throttle + x*states.at(after).throttle;
  return ret;
}
 
void LapRecord::dumpTo(std::ostream& out) const
{
  out << " " << lap_start << " ";
  out << states.size() << " ";
  for (const_iterator it = begin(); it != end(); ++it)
  {
    it->dumpTo(out);
    out << " ";
  }
}

void LapRecord::loadFrom(std::istream &in)
{
  in >> lap_start;
  unsigned int count;
  in >> count;
  ContainerT tmp;
  tmp.reserve(count);
  for (unsigned int i=0; i<count; ++i)
  {
    top10::physX::Kart::State s;
    s.loadFrom(in);
    tmp.push_back(s);
  }
  states.swap(tmp);
}

}
}
