#include "ListPick.hh"
#include "../util/AutoSet.hh"

#include <algorithm>
#include <QtGui/QListWidget>
#include <QtGui/QListWidgetItem>

using top10::util::AutoSet;

namespace top10
{
  namespace qt
  {
    ListPickWidget::ListPickWidget(QWidget *parent)
      : QWidget(parent),
      m_emit(true)
    {
      setupUi(this);
    }



    /*
     * Public slots
     */

    bool ListPickWidget::insertItem(const QString& item)
    {
      QList< QListWidgetItem* > items = m_excluded->findItems(item, Qt::MatchExactly);

      if (items.size() == 0)
      {
	const int pos = m_excluded->currentRow() != -1 ? m_excluded->currentRow() : m_excluded->count();
	m_excluded->insertItem(pos, item);
	
	if (m_emit)
	  emit itemInserted(pos, item);

	return true;
      }

      return false;
    }



    bool ListPickWidget::includeItem(const QString& item, bool insert_last)
    {
      QListWidgetItem* entry = takeItem(item, m_excluded);

      if (entry)
      {
	const int pos = insert_last ? m_included->count() : m_included->currentRow();
	m_included->insertItem(pos, entry);

	if (m_emit)
	  emit itemIncluded(pos, item);

	return true;
      }
      return false;
    }




    bool ListPickWidget::excludeItem(const QString& item)
    {
      QListWidgetItem* entry = takeItem(item, m_included);

      if (entry)
      {
	const int pos = m_excluded->currentRow();
	m_excluded->insertItem(pos, entry);

	if (m_emit)
	  emit itemExcluded(pos, item);

	return true;
      }
      return false;
    }




    void ListPickWidget::clear()
    {
      m_excluded->clear();
      m_included->clear();

      if (m_emit)
	emit cleared();
    }



    void ListPickWidget::clearIncluded()
    {
      for (int pos = 0; pos < m_included->count(); ++pos)
      {
	QListWidgetItem* entry = m_included->takeItem(pos);
	m_excluded->addItem(entry);
      }

      if (m_emit)
	emit includedCleared();
    }



    bool ListPickWidget::removeItem(const QString& item)
    {
      bool ret = removeItem(item, m_excluded) || removeItem(item, m_included);
      
      if (ret && m_emit)
	emit itemRemoved(item);

      return ret;
    }



    /*
     * Public methods
     */

    bool ListPickWidget::insertItemNoEmit(const std::string& item)
    {
      AutoSet<bool> auto_set(&m_emit, false);
      return insertItem(QString(item.c_str()));
    }



    bool ListPickWidget::includeItemNoEmit(const std::string& item, bool insert_last)
    {
      AutoSet<bool> auto_set(&m_emit, false);
      return includeItem(QString(item.c_str()), insert_last);
    }



    bool ListPickWidget::excludeItemNoEmit(const std::string& item)
    {
      AutoSet<bool> auto_set(&m_emit, false);
      return excludeItem(QString(item.c_str()));
    }



    void ListPickWidget::clearNoEmit()
    {
      AutoSet<bool> auto_set(&m_emit, false);
      clear();
    }



    void ListPickWidget::clearIncludedNoEmit()
    {
      AutoSet<bool> auto_set(&m_emit, false);
      clearIncluded();
    }



    bool ListPickWidget::removeItemNoEmit(const std::string& item)
    {
      AutoSet<bool> auto_set(&m_emit, false);
      const QString s(item.c_str());

      return removeItem(s, m_excluded) || removeItem(s, m_included);
    }



    /*
     * Private slots
     */

    void ListPickWidget::on_m_add_clicked()
    {
      QString entry;

      if (m_excluded->currentItem())
	entry = m_excluded->currentItem()->text();
      else if (m_excluded->count() > 0)
	entry = m_excluded->item(0)->text();
      
      if (!entry.isEmpty())
	includeItem(entry);
    }




    void ListPickWidget::on_m_remove_clicked()
    {
      QString entry;
      
      if (m_included->currentItem())
       entry = m_included->currentItem()->text();
      else if (m_included->count() > 0)
	entry = m_included->item(0)->text();
      
      if (!entry.isEmpty())
	excludeItem(entry);
    }


    
    /*
     * Private
     */

    bool ListPickWidget::removeItem(const QString& item, QListWidget* widget)
    {
      QListWidgetItem* removed = takeItem(item, widget);

      if (removed)
      {
	delete removed;
	return true;
      }

      return false;  
    }




     QListWidgetItem*
       ListPickWidget::takeItem(const QString& item, QListWidget* widget)
    {
      int pos;
      for (pos=0; pos < widget->count() && widget->item(pos)->text() != item; ++pos);

      if (pos >= 0 && pos < widget->count())
      {
	return widget->takeItem(pos);
      }

      return 0;  
    }

  }
}
