#include "ComboBoxEdit.hh"
#include "../util/AutoSet.hh"
#include <QtGui/QLineEdit>

using top10::util::AutoSet;

namespace top10
{
  namespace qt
  {

    ComboBoxEdit::ComboBoxEdit(QWidget *parent)
      : QWidget(parent),
      m_emit(true)
    {
      setupUi(this);
      connect( m_items->lineEdit(), SIGNAL(returnPressed()), this, SLOT(on_m_items_edited()) );
    }



    /*
     * Public methods.
     */

    bool ComboBoxEdit::insertItemNoEmit(const std::string& item)
    {
      AutoSet<bool> auto_set(&m_emit, false);
      return insertItem(QString(item.c_str()));
    }



    bool ComboBoxEdit::removeItemNoEmit(const std::string& item)
    {
      AutoSet<bool> auto_set(&m_emit, false);
      return removeItem(QString(item.c_str()));
    }



    bool ComboBoxEdit::setCurrentItemNoEmit(const std::string& item)
    {
      AutoSet<bool> auto_set(&m_emit, false);
      return setCurrentItem(QString(item.c_str()));
    }



    bool ComboBoxEdit::setCurrentNameNoEmit(const std::string& name)
    {
      AutoSet<bool> auto_set(&m_emit, false);
      return setCurrentName(QString(name.c_str()));
    }



    void ComboBoxEdit::clearNoEmit()
    {
      AutoSet<bool> auto_set(&m_emit, false);
      clear();
    }



    /*
     * Public slots.
     */

    bool ComboBoxEdit::insertItem(const QString& item)
    {
      bool found = m_items->findText(item) != -1;
      if (found)
	return false;

      m_items->insertItem(m_items->count(), item);
      m_items->setCurrentIndex(m_items->count() -1);

      if (m_emit)
	emit itemInserted(item);

      return true;
    }



    bool ComboBoxEdit::removeItem(const QString& item)
    {
      int pos = m_items->findText(item);
      if (pos != -1)
      {
	m_items->removeItem(pos);

	if (m_emit)
	  emit itemRemoved(item);

	return true;
      }

      return false;
    }




    bool ComboBoxEdit::setCurrentItem(const QString& item)
    {
      int pos = m_items->findText(item);
      if (pos != -1)
      {
	m_items->setCurrentIndex(pos);

	if (m_emit)
	  emit itemSelected(item);

	return true;
      }

      return false;
    }



    bool ComboBoxEdit::setCurrentName(const QString& name)
    {
      if (name == m_items->currentText())
	return true;

      int pos = m_items->currentIndex();
      m_items->removeItem(pos);
      m_items->insertItem(pos, name);

      if (m_emit)
	emit itemRenamed(name);

      return true;
    }



    void ComboBoxEdit::clear()
    {
      m_items->clear();

      if (m_emit)
	emit cleared();
    }



    /*
     * Private slots.
     */

    void ComboBoxEdit::on_m_new_clicked()
    {
      bool status = insertItem("<New>");
    }



    void ComboBoxEdit::on_m_delete_clicked()
    {
      int pos = m_items->currentIndex();

      if (pos >= 0 && pos < m_items->count())
	removeItem(m_items->currentText());
    }



    void ComboBoxEdit::on_m_items_currentIndexChanged(const QString& item)
    {
      if (m_emit)
	emit itemSelected(item);
    }



    void ComboBoxEdit::on_m_items_edited()
    {
      if (m_emit)
	emit itemRenamed( m_items->currentText() );
    }

  }
}
