/*
  Top10, a racing simulator
  Copyright (C) 2000-2007  Johann Deneux
  
  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.
  
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
  
  Authors can be contacted at following electronic addresses:
  Johann Deneux: johann.deneux@gmail.com
*/

#include "physX/WheelState.hh"
#include <iostream>

void testBounce(const top10::physX::WheelProperties* props, const double dt)
{
  top10::physX::Surface surf = top10::physX::SurfaceConcrete();
  top10::math::Plane ground;
  top10::physX::SurfacePatch patch(ground, surf);

  top10::physX::WheelState state(props);
  state.getRigidBody()->setTranslation(top10::math::Vector(0.0, 1.0, 0.0));

  const top10::math::Vector gravity(0.0, -9.81*props->getMass(), 0.0);

  double old_energy = 0.0;
  int iter = 0;
  bool stable = false;
  for (double t=0.0; !stable && t<100.0; t+=dt)
  {
    state.getRigidBody()->clearForces();
    state.getRigidBody()->applyForceAt(state.getRigidBody()->getCenter(), gravity);
    state.computeGroundForces(patch, -gravity.y);
    state.getRigidBody()->integrate(t, dt);

    const double energy = state.getRigidBody()->getSpeed().size2() + state.getRigidBody()->getWSpeed().size2() + state.getRigidBody()->getTranslation().y;
    
    if (energy < old_energy && (old_energy-energy) / dt < 1e-4)
    {
      std::cout << "Stable after " << t <<std::endl;
      stable = true;
    }

    if (iter == 50)
    {
      std::cout << state.getRigidBody()->getTranslation() << " | ";
      std::cout << state.getRigidBody()->getSpeed() << " | ";
      std::cout << energy << std::endl;
      iter = 0;
    }
    else
      ++iter;

    old_energy = energy;
  }
}

double testStraight(const top10::physX::WheelProperties* props, const double dt, double torque, double speed)
{
  std::cout << "\nExperiment: torque = " << torque << std::endl;

  top10::physX::Surface surf = top10::physX::SurfaceConcrete();
  top10::math::Plane ground;
  top10::physX::SurfacePatch patch(ground, surf);

  top10::physX::WheelState state(props);
  state.getRigidBody()->setTranslation(top10::math::Vector(0.0, props->getRadius(), 0.0));

  const top10::math::Vector gravity(0.0, -9.81*props->getMass(), 0.0);

  double avg_long_grip = 0.0;
  double n_samples = 0.0;

  double avg_slip_ratio = 0.0;
  double n_ratios = 0.0;
  double long_grip = 0.0;

  int iter = 0;
  bool speed_reached = false;
  double t=0.0;
  for (; !speed_reached && t<300.0; t+=dt)
  {
    state.getRigidBody()->clearForces();
    state.getRigidBody()->applyForceAt(state.getRigidBody()->getCenter(), gravity);
    state.getRigidBody()->applyTorque(top10::math::Vector(0.0, 0.0, torque));
    top10::physX::WheelProperties::FrictionData data;
    state.computeGroundForces(patch, -gravity.y, &data);
    state.getRigidBody()->integrate(t, dt);

    long_grip += data.m_long_grip;

    if (fabs(data.m_slip_ratio) < 1e12)
    {
      avg_slip_ratio = n_ratios/(1.0 + n_ratios) * avg_slip_ratio + fabs(data.m_slip_ratio)/(1.0 + n_ratios);
      n_ratios += 1.0;
    }

    avg_long_grip = n_samples/(1.0 + n_samples) * avg_long_grip + data.m_long_grip/(1.0 + n_samples);
    n_samples += 1.0;

    if (false && iter == 50)
    {
      std::cout << state.getRigidBody()->getTranslation() << " | ";
      std::cout << state.getRigidBody()->getSpeed() << std::endl;
      std::cout << "Grip: " << long_grip / 50.0 << std::endl;
      iter = 0;
      long_grip = 0.0;
    }
    else
      ++iter;

    if (-state.getRigidBody()->getSpeed().x >= speed)
    {
      std::cout << "Speed reached after " << t << "s at " << state.getRigidBody()->getTranslation().x << "m" << std::endl;
      std::cout << "Torque x Time = " << torque*t << std::endl;
      std::cout << "Average grip: " << avg_long_grip << std::endl;
      std::cout << "Average slip ratio: " << avg_slip_ratio << std::endl;
      speed_reached = true;
    }
  }

  if (!speed_reached)
  {
    std::cout << "Speed NOT reached after " << t <<"s at " << -state.getRigidBody()->getTranslation().x << "m" << std::endl;
    return 1e6;
  }

  std::cout << "Slip quality: " << n_ratios / n_samples << std::endl;

  return t;
}

void searchOptimalTorque(const top10::physX::WheelProperties* props, const double dt, double speed)
{
  // torques
  double T[5];

  // times
  double t[5];

  T[0] = 10.0;
  T[4] = 125.0;

  do
  {
    std::cout << "\n*** Search iter " << T[0] << ", " << T[4] << " ***\n";

    T[2] = 0.5*(T[0]+T[4]);
    T[1] = 0.5*(T[0]+T[2]);
    T[3] = 0.5*(T[2]+T[4]);

    double t_min = 1e7;
    int idx_min = -1;
    for (int i=0; i<4; ++i)
    {
      t[i] = testStraight(props, dt, T[i], speed);
      if (t[i] < t_min)
      {
	t_min = t[i];
	idx_min = i;
      }
    }

    assert( idx_min > -1 );

    if (idx_min == 0)
    {
      T[0] = 2.0 * T[0] - T[1];
      T[4] = T[1];
    }
    else if (idx_min == 4)
    {
      T[4] = 2.0 * T[4] - T[3];
      T[0] = T[3];
    }
    else
    {
      assert(idx_min >= 1 && idx_min <= 3);
      T[0] = T[idx_min -1];
      T[4] = T[idx_min +1];
    }

    if (T[0] < 0.0)
      T[0] = 0.0;

  } while (T[4] - T[0] >= 0.5);

  std::cout << "\nOptimal torque is " << T[2] << " (" << t[2] << "s)" << std::endl;
}

int main(int argc, char** argv)
{
  top10::util::Ref<top10::physX::WheelProperties> props;
  props = new top10::physX::WheelProperties;
  props->setMass(40.0);
  props->setRadius(0.13);
  props->setInertia(0.1);
  props->setInertiaVert(0.05);
  
  props->insertGripLong(0.0, 0.0);
  props->insertGripLong(0.01, 2.0);
  props->insertGripLong(0.05, 2.0);
  props->insertGripLong(0.07, 1.7);
  props->insertGripLong(0.2, 1.5);

  props->insertGripLatDegrees(0.0, 0.0);
  props->insertGripLatDegrees(8.0, 2.0);
  props->insertGripLatDegrees(20.0, 1.6);

  testBounce(props.getPtr(), 1e-3);
  testBounce(props.getPtr(), 1e-2);
  
  searchOptimalTorque(props.getPtr(), 1e-3, 80.0 * 1000.0 / 3600.0);

  return 0;
}
