/*
  Top10, a racing simulator
  Copyright (C) 2000-2007  Johann Deneux
  
  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.
  
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
  
  Authors can be contacted at following electronic addresses:
  Johann Deneux: johann.deneux@gmail.com
*/

#include "physX/RigidBodyState.hh"
#include "physX/LinearSpringProperties.hh"
#include "physX/AngularSpringProperties.hh"

void testRest(const top10::physX::RigidBodyProperties* props)
{
  std::cout << "testRest: ";
  top10::physX::RigidBodyState state(props);

  const double dt = 0.01;
  for (double t=0.0; t < 10.0; t += dt)
  {
    state.integrate(t, dt);
  }

  if (state.getTranslation() == top10::math::Vector(0.0, 0.0, 0.0) &&
    state.getOrient() == top10::math::Identity3())
  {
    std::cout << "OK";
  }
  else
    std::cout << "Failed";

  std::cout << std::endl;
}

void testForceBalance(const top10::physX::RigidBodyProperties* props)
{
  std::cout << "testForceBalance: ";
  
  top10::physX::RigidBodyState state(props);

  const top10::math::Vector center = props->getCenter();

  const double dt = 0.01;
  for (double t=0.0; t < 10.0; t += dt)
  {
    state.clearForces();
    state.applyForceAt(center + top10::math::Vector(2.0, 0.0, 0.0), top10::math::Vector(0.0, 1.0, 0.0));
    state.applyForceAt(center + top10::math::Vector(-2.0, 0.0, 0.0), top10::math::Vector(0.0, 1.0, 0.0));
    state.applyForceAt(center + top10::math::Vector(0.0, 0.0, 2.0), top10::math::Vector(0.0, 1.0, 0.0));
    state.applyForceAt(center + top10::math::Vector(0.0, 0.0, -2.0), top10::math::Vector(0.0, 1.0, 0.0));
    state.applyForceAt(center, top10::math::Vector(0.0, -4.0, 0.0));
    state.integrate(t, dt);
  }

  if (state.getTranslation().size2() < 0.001 &&
    state.getSpeed().size2() < 0.001 &&
    state.getWSpeed().size2() < 0.001)
  {
    std::cout << "OK";
  }
  else
    std::cout << "Failed";

  std::cout << std::endl;
}

void testForceCentral(const top10::physX::RigidBodyProperties* props)
{
  std::cout << "testForceCentral: ";
  
  top10::physX::RigidBodyState state(props);

  const top10::math::Vector center = props->getCenter();
  const top10::math::Vector force(-1.0, -4.0, 2.0);
  const double dt = 0.01;
  for (double t=0.0; t < 10.0; t += dt)
  {
    state.clearForces();
    state.applyForceAt(center, force);
    state.integrate(t, dt);
  }

  if ((state.getTranslation() ^ force).size2() < 0.001 &&
    (state.getSpeed() ^ force).size2() < 0.001 &&
    state.getWSpeed().size2() < 0.001 &&
    state.getTranslation() * force > 0.0 &&
    state.getSpeed() * force > 0.0)
  {
    std::cout << "OK";
  }
  else
    std::cout << "Failed";

  std::cout << std::endl;
}

void testForceRotate(const top10::physX::RigidBodyProperties* props)
{
  std::cout << "testForceRotate: ";
  
  top10::physX::RigidBodyState state(props);

  const top10::math::Vector center = props->getCenter();
  const top10::math::Vector force(-1.0, -4.0, 2.0);
  const double dt = 0.01;
  for (double t=0.0; t < 10.0; t += dt)
  {
    state.clearForces();
    state.applyForceAt(center + top10::math::Vector(1.0, 0.0, 1.0), force);
    state.applyForceAt(center + top10::math::Vector(-1.0, 0.0, -1.0), -force);
    state.integrate(t, dt);
  }

  const top10::math::Vector torque = top10::math::Vector(1.0, 0.0, 1.0) ^ force;

  if (state.getTranslation().size2() < 0.001 &&
    state.getSpeed().size2() < 0.001 &&
    state.getWSpeed() * torque > 0.0)
  {
    std::cout << "OK";
  }
  else
    std::cout << "Failed";

  std::cout << std::endl;
}

void testTorqueRotate(const top10::physX::RigidBodyProperties* props)
{
  std::cout << "testTorqueRotate: ";
  
  top10::physX::RigidBodyState state(props);

  const top10::math::Vector center = props->getCenter();
  const top10::math::Vector force(-1.0, -4.0, 2.0);
  const top10::math::Vector torque = top10::math::Vector(1.0, 0.0, 1.0) ^ force;

  const double dt = 0.01;
  for (double t=0.0; t < 10.0; t += dt)
  {
    state.clearForces();
    state.applyTorque(torque);
    state.integrate(t, dt);
  }

  if (state.getTranslation().size2() < 0.001 &&
    state.getSpeed().size2() < 0.001 &&
    state.getWSpeed() * torque > 0.0)
  {
    std::cout << "OK";
  }
  else
    std::cout << "Failed";

  std::cout << std::endl;
}

void testForceSpring(const top10::physX::RigidBodyProperties* props, double dt)
{
  std::cout << "testForceSpring "<<dt<<": ";
  
  top10::physX::RigidBodyState state(props);

  const double stiffness = 1.5e4;
  const double damping = 1e3;
  top10::physX::LinearSpringProperties spring;
  spring.setStiffness(stiffness);
  spring.setDamping(damping);

  const top10::math::Vector attach_pt = props->getCenter() /*+ top10::math::Vector(0.0, 1.0, 0.0)*/;
  const top10::math::Vector fixed_pt = attach_pt + top10::math::Vector(0.1, 0.1, 0.1);

  int iter = 0;
  bool stable = false;
  double old_energy_lin = 0.0;
  double old_energy_rot = 0.0;
  for (double t=0.0; !stable && t < 10.0; t += dt)
  {
    state.clearForces();
    top10::math::Vector pt = state.localToGlobal(attach_pt);
    top10::math::Vector force = spring.getForce(pt-fixed_pt, state.getSpeedAtL(attach_pt));
    state.applyForceAt(pt, force);
    state.integrate(t, dt);

    double energy_lin = state.getSpeed().size2() + (pt - fixed_pt).size();
    double energy_rot = state.getWSpeed().size2();

    if (energy_lin + energy_rot < 1e-5 && energy_lin + energy_rot < old_energy_lin + old_energy_rot)
    {
      std::cout << "Stabilized after " << t <<"s." << std::endl;
      stable = true;
    }

    if (false && iter == 10)
    {
      iter = 0;
      std::cout << "Energy = " << energy_lin << ", " << energy_rot << "\n";
      std::cout << "Dist = " << (pt-fixed_pt).size() << " Force = " << force << std::endl;
    }
    else
      ++iter;

    old_energy_lin = energy_lin;
    old_energy_rot = energy_rot;
  }

  if (stable)
  {
    std::cout << "OK";
  }
  else
    std::cout << "Failed";

  std::cout << std::endl;
}

top10::math::Vector computeTorqueSpring(top10::math::Vector v1, top10::math::Vector v2, double stiffness)
{
  top10::math::Vector ret;

  double s1 = v1.size();
  double s2 = v2.size();

  if (s1 >= SMALL_VECTOR && s2 >= SMALL_VECTOR)
  {
    v1 /= s1;
    v2 /= s2;

    ret = stiffness * (v2 ^ v1);
  }

  return ret;
}

void testTorqueSpring(const top10::physX::RigidBodyProperties* props, const double dt)
{
  std::cout << "testTorqueSpring "<<dt<<": ";
  
  top10::physX::RigidBodyState state(props);

  const double stiffness = 1e4;
  const double damping = 1e3;
  top10::physX::AngularSpringProperties spring;
  spring.setStiffness(stiffness);
  spring.setDamping(damping);

  const top10::math::Vector attach_pt = top10::math::Vector(0.0, 1.0, 0.0);
  top10::math::Vector fixed_pt = top10::math::Vector(0.7, 0.7, 0.0);
  fixed_pt /= fixed_pt.size();

  int iter = 0;
  bool stable = false;
  double old_energy_rot = 0.0;
  for (double t=0.0; !stable && t < 10.0; t += dt)
  {
    state.clearForces();

    top10::math::Matrix3 orient = state.getOrient();
    top10::math::Vector pt = orient * attach_pt;

    top10::math::Vector normal = fixed_pt^pt;
    double normal_size = normal.size();
    if (normal_size >= SMALL_VECTOR)
      normal /= normal_size;

    top10::math::Vector w_speed = orient * state.getWSpeed();

    top10::math::Vector torque = computeTorqueSpring(fixed_pt, pt, stiffness) - damping*(normal*(normal*w_speed));
//    top10::math::Vector torque = spring.getTorque(fixed_pt, pt, w_speed);

    state.applyTorque(torque);
    state.integrate(t, dt);

    double energy_rot = w_speed.size2() + normal_size;

    if (energy_rot < 1e-5 && energy_rot < old_energy_rot)
    {
      std::cout << "Stabilized after " << t <<"s." << std::endl;
      stable = true;
    }

    if (false && iter == 50)
    {
      iter = 0;
      std::cout << "Energy = " << energy_rot << "\n";
      std::cout << "Angle = " << normal_size << " Torque = " << torque << std::endl;
    }
    else
      ++iter;

    old_energy_rot = energy_rot;
  }

  if (stable)
  {
    std::cout << "OK";
  }
  else
    std::cout << "Failed";

  std::cout << std::endl;
}

int main(int argc, char** argv)
{
  // Create the body properties
  top10::util::Ref<top10::physX::RigidBodyProperties> body_properties;
  body_properties = new top10::physX::RigidBodyProperties;

  // Add masses
  body_properties->addMass( top10::math::Vector(0.0, 0.0, 0.0), 10.0 );
  body_properties->addMass( top10::math::Vector(-1.0, 0.0, 0.0), 5.0 );
  body_properties->addMass( top10::math::Vector(0.0, -5.0, 0.0), 5.0 );

  body_properties->update();
  testRest(body_properties.getPtr());
  testForceBalance(body_properties.getPtr());
  testForceCentral(body_properties.getPtr());
  testForceRotate(body_properties.getPtr());
  testTorqueRotate(body_properties.getPtr());

  testForceSpring(body_properties.getPtr(), 5e-5);
  testForceSpring(body_properties.getPtr(), 1e-4);
  testForceSpring(body_properties.getPtr(), 1e-3);
  testForceSpring(body_properties.getPtr(), 0.0025);
  testForceSpring(body_properties.getPtr(), 0.01);
  testForceSpring(body_properties.getPtr(), 0.05);
  testForceSpring(body_properties.getPtr(), 0.1);

  testTorqueSpring(body_properties.getPtr(), 5e-5);
  testTorqueSpring(body_properties.getPtr(), 1e-4);
  testTorqueSpring(body_properties.getPtr(), 1e-3);
  testTorqueSpring(body_properties.getPtr(), 0.0025);
  testTorqueSpring(body_properties.getPtr(), 0.01);
  testTorqueSpring(body_properties.getPtr(), 0.05);
  testTorqueSpring(body_properties.getPtr(), 0.1);

  return 0;
}
