/*
  Top 10, a racing simulator
  Copyright (C) 2003-2007  Johann Deneux
  
  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.
  
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
  
  Authors can be contacted at following electronic addresses:
  Johann Deneux: johann.deneux@gmail.com
*/

#include "WheelState.hh"

namespace top10
{
  namespace physX
  {

    WheelState::WheelState(const WheelProperties* props)
      : m_props(props),
      m_rigid_state( new RigidBodyProperties )
    {
      RigidBodyProperties* rigid_props = new RigidBodyProperties;

      // Set a few masses to match the inertia of the wheel
      const double d1 = sqrt(1.5 * m_props->getInertia() / m_props->getMass());
      const double d3 = sqrt((m_props->getInertiaVert() - 0.25*m_props->getInertia())*3.0/m_props->getMass());
      const double m3 = m_props->getMass();

      rigid_props->addMass( top10::math::Vector(d1, 0.0, 0.0), m3 );
      rigid_props->addMass( top10::math::Vector(-d1, 0.0, 0.0), m3 );
      rigid_props->addMass( top10::math::Vector(0.0, d1, 0.0), m3 );
      rigid_props->addMass( top10::math::Vector(0.0, -d1, 0.0), m3 );
      rigid_props->addMass( top10::math::Vector(0.0, 0.0, d3), m3 );
      rigid_props->addMass( top10::math::Vector(0.0, 0.0, -d3), m3 );
      rigid_props->update();

      m_rigid_props = rigid_props;

      RigidBodyState tmp(rigid_props);
      m_rigid_state = tmp;
    }

    RigidBodyState* WheelState::getRigidBody()
    {
      return &m_rigid_state;
    }

    void WheelState::computeGroundForces(const SurfacePatch& surface, double load, WheelProperties::FrictionData* out_data)
    {
      const double dist = m_rigid_state.getTranslation() * surface.normal - surface.d;

      if (dist > m_props->getRadius())
	return;

      const top10::math::Vector e3 = m_rigid_state.getOrient() * top10::math::Vector(0.0, 0.0, 1.0);
      const top10::math::Vector friction = 
	surface.grip * m_props->computeFriction( load, -surface.normal,
	                                         m_rigid_state.getWSpeed() * e3, e3,
	                                         m_rigid_state.getSpeed(), out_data );

      const top10::math::Vector reaction = 
	m_props->computeReaction( load, -surface.normal, dist, m_rigid_state.getSpeed() );

      const top10::math::Vector pos = m_rigid_state.getTranslation() - dist * surface.normal;
      m_rigid_state.applyForceAt( pos, friction + reaction );
    }
  }
}
