/*
  Top 10, a racing simulator
  Copyright (C) 2003-2007  Johann Deneux
  
  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.
  
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
  
  Authors can be contacted at following electronic addresses:
  Johann Deneux: johann.deneux@gmail.com
*/

#ifndef TOP10_PHYSX_WHEEL_PROPS_HH
#define TOP10_PHYSX_WHEEL_PROPS_HH

#include "util/XmlDumpable.hh"
#include "util/RefCount.hh"
#include "math/Vertex.hh"
#include "math/Curve.hh"

namespace top10
{
  namespace physX
  {
    class WheelProperties: public top10::util::RefCount, public top10::util::XmlDumpable
    {
    public:
      static const char* NODE_NAME;

      //! Optional parameter to computFriction, useful for debugging and recording.
      struct FrictionData
      {
	FrictionData(): m_lat_grip(0.0), m_long_grip(0.0), m_slip_angle(0.0), m_slip_ratio(0.0) {}

	double m_lat_grip;
	double m_long_grip;
	double m_slip_angle;
	double m_slip_ratio;
      };

    public:
      WheelProperties();
      void clearState();

      void setMass(double);
      void setRadius(double);
      void setInertia(double);
      void setInertiaVert(double);

      //! Insert a point in the longitudinal grip curve.
      void insertGripLong(double slip_ratio, double grip);

      //! Inert a point in the lateral grip curve.
      /*!
	\param slip_angle Square of the sin() of the slip angle.
	\param grip Grip coefficient.
	*/
      void insertGripLat(double slip_angle, double grip);

      //! Insert a point in the lateral grip curve.
      /*!
	\param slip_angle Slip angle, in degrees.
	\param grip Grip coefficient.
      */
      void insertGripLatDegrees(double slip_angle, double grip);

      double getMass() const;
      double getInertia() const;
      double getInertiaVert() const;
      double getRadius() const;


      //! Compute the friction force
      /*!
	\param load Load on the wheel (N)
	\param down Unit vector from the center of the wheel to the contact point with the ground.
	\param w_speed Angular speed (rad/s)
	\param w_dir Unit vector, axis of rotation of the wheel
	\param speed Speed of the wheel wrt to the ground.
      */
      top10::math::Vector computeFriction(double load, top10::math::Vector down,
	                                  double w_speed, top10::math::Vector w_dir,
					  top10::math::Vector speed,
					  FrictionData* out_data =0) const;

      top10::math::Vector computeReaction(double load, top10::math::Vector down,
	                                  double dist,
	                                  top10::math::Vector speed) const;

    private:
      double getGripLong(double) const;
      double getGripLat(double) const;

    protected:
      int loadXml(const TiXmlElement*);
      int saveXml(TiXmlElement*) const;

    private:

      //! Outer radius of the tyre.
      double m_radius;

      //! Mass of the wheel.
      double m_mass;

      //! Inertia around the axis of the wheel.
      double m_inertia;

      //! Inertia around the vertical axis.
      double m_inertia_y;

      //! Grip curve: mapping from slip ratio to grip coefficient.
      top10::math::Curve2D m_grip_long;

      //! Lateral grip curve: mapping from sin(slip angle (in rad))^2 to grip coefficient.
      top10::math::Curve2D m_grip_lat;
    };
  }
}

#endif
